/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.common.transfer;

import me.shedaniel.rei.api.common.entry.InputIngredient;
import me.shedaniel.rei.api.common.transfer.ItemRecipeFinder;
import me.shedaniel.rei.api.common.transfer.info.stack.SlotAccessor;
import net.minecraft.class_1263;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import java.util.List;
import java.util.function.Predicate;

public class NewInputSlotCrafter<T extends class_1703, C extends class_1263> extends InputSlotCrafter<T, C> {
    protected final List<SlotAccessor> inputSlots;
    protected final List<SlotAccessor> inventorySlots;
    protected final List<InputIngredient<class_1799>> inputs;
    
    public NewInputSlotCrafter(T container, List<SlotAccessor> inputSlots, List<SlotAccessor> inventorySlots, List<InputIngredient<class_1799>> inputs) {
        super(container);
        this.inputSlots = inputSlots;
        this.inventorySlots = inventorySlots;
        this.inputs = inputs;
    }
    
    @Override
    protected Iterable<SlotAccessor> getInputSlots() {
        return this.inputSlots;
    }
    
    @Override
    protected Iterable<SlotAccessor> getInventorySlots() {
        return this.inventorySlots;
    }
    
    @Override
    protected List<InputIngredient<class_1799>> getInputs() {
        return this.inputs;
    }
    
    @Override
    protected void populateRecipeFinder(ItemRecipeFinder recipeFinder) {
        for (SlotAccessor slot : getInventorySlots()) {
            recipeFinder.addNormalItem(slot.getItemStack());
        }
    }
    
    @Override
    protected void markDirty() {
        player.method_31548().method_5431();
        container.method_34252();
    }
    
    @Override
    protected void cleanInputs() {
        for (SlotAccessor slot : getInputSlots()) {
            class_1799 stackToReturn = slot.getItemStack();
            if (!slot.allowModification(player)) {
                throw new IllegalStateException("Slot " + slot + " is not modifiable!");
            }
            if (!stackToReturn.method_7960()) {
                for (; !(stackToReturn = slot.getItemStack()).method_7960(); slot.takeStack(1)) {
                    class_1799 stackToInsert = stackToReturn.method_7972();
                    stackToInsert.method_7939(1);
                    
                    if (!getDumpHandler().test(stackToInsert)) {
                        throw new IllegalStateException("rei.rei.no.slot.in.inv");
                    }
                }
            }
        }
    }
    
    private Predicate<class_1799> getDumpHandler() {
        return (stackToDump) -> {
            Iterable<SlotAccessor> inventoryStacks = getInventorySlots();
            SlotAccessor occupiedSlotWithRoomForStack = getOccupiedSlotWithRoomForStack(stackToDump, inventoryStacks);
            SlotAccessor emptySlot = getEmptySlot(inventoryStacks);
            
            SlotAccessor nextSlot = occupiedSlotWithRoomForStack == null ? emptySlot : occupiedSlotWithRoomForStack;
            if (nextSlot == null) {
                return false;
            }
            
            class_1799 stack = stackToDump.method_7972();
            stack.method_7939(nextSlot.getItemStack().method_7947() + stack.method_7947());
            nextSlot.setItemStack(stack);
            return true;
        };
    }
    
    static SlotAccessor getOccupiedSlotWithRoomForStack(class_1799 stack, Iterable<SlotAccessor> inventoryStacks) {
        for (SlotAccessor inventoryStack : inventoryStacks) {
            if (canStackAddMore(inventoryStack.getItemStack(), stack)) {
                return inventoryStack;
            }
        }
        
        return null;
    }
    
    static SlotAccessor getEmptySlot(Iterable<SlotAccessor> inventoryStacks) {
        for (SlotAccessor inventoryStack : inventoryStacks) {
            if (inventoryStack.getItemStack().method_7960()) {
                return inventoryStack;
            }
        }
        
        return null;
    }
    
    static boolean canStackAddMore(class_1799 existingStack, class_1799 stack) {
        return !existingStack.method_7960() && class_1799.method_31577(existingStack, stack) && existingStack.method_7946() && existingStack.method_7947() + stack.method_7947() <= existingStack.method_7914();
    }
}
