/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.registry.screen;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import dev.architectury.event.CompoundEventResult;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.gui.config.DisplayPanelLocation;
import me.shedaniel.rei.api.client.gui.drag.DraggableStackProvider;
import me.shedaniel.rei.api.client.gui.drag.DraggableStackVisitor;
import me.shedaniel.rei.api.client.gui.drag.component.DraggableComponentProvider;
import me.shedaniel.rei.api.client.gui.drag.component.DraggableComponentProviderWidget;
import me.shedaniel.rei.api.client.gui.drag.component.DraggableComponentVisitor;
import me.shedaniel.rei.api.client.gui.drag.component.DraggableComponentVisitorWidget;
import me.shedaniel.rei.api.client.gui.widgets.Widgets;
import me.shedaniel.rei.api.client.overlay.ScreenOverlay;
import me.shedaniel.rei.api.client.plugins.REIClientPlugin;
import me.shedaniel.rei.api.client.registry.screen.*;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.registry.ReloadStage;
import me.shedaniel.rei.api.common.util.CollectionUtils;
import me.shedaniel.rei.api.common.util.EntryStacks;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.screen.AbstractDisplayViewingScreen;
import me.shedaniel.rei.impl.common.InternalLogger;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1041;
import net.minecraft.class_1269;
import net.minecraft.class_1703;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_437;
import net.minecraft.class_465;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;

@ApiStatus.Internal
@Environment(EnvType.CLIENT)
public class ScreenRegistryImpl implements ScreenRegistry {
    private Multimap<Class<? extends class_437>, ClickArea<?>> clickAreas = HashMultimap.create();
    private List<DraggableComponentProvider<class_437, Object>> draggableProviders = new CopyOnWriteArrayList<>();
    private List<DraggableComponentVisitor<class_437>> draggableVisitors = new CopyOnWriteArrayList<>();
    private List<FocusedStackProvider> focusedStackProviders = new CopyOnWriteArrayList<>();
    private List<OverlayDecider> deciders = new CopyOnWriteArrayList<>();
    private Map<Class<?>, List<OverlayDecider>> cache = new HashMap<>();
    private ExclusionZones exclusionZones = new ExclusionZonesImpl();
    private OverlayRendererProvider lastRendererProvider = null;
    private final ThreadLocal<Class<? extends class_437>> tmpScreen = new ThreadLocal<>();
    
    @Override
    public ReloadStage getStage() {
        return ReloadStage.START;
    }
    
    @Override
    public void acceptPlugin(REIClientPlugin plugin) {
        plugin.registerScreens(this);
        plugin.registerExclusionZones(exclusionZones());
    }
    
    @Override
    public <R extends class_437> List<OverlayDecider> getDeciders(R screen) {
        if (screen == null) return Collections.emptyList();
        Class<? extends class_437> screenClass = screen.getClass();
        List<OverlayDecider> possibleCached = cache.get(screenClass);
        if (possibleCached != null) {
            return possibleCached;
        }
        
        tmpScreen.set(screenClass);
        List<OverlayDecider> deciders = CollectionUtils.filterToList(this.deciders, this::filterResponsible);
        cache.put(screenClass, deciders);
        tmpScreen.remove();
        return deciders;
    }
    
    private boolean filterResponsible(OverlayDecider handler) {
        return handler.isHandingScreen(tmpScreen.get());
    }
    
    @Override
    public List<OverlayDecider> getDeciders() {
        return Collections.unmodifiableList(deciders);
    }
    
    @Override
    public <T extends class_437> Rectangle getScreenBounds(T screen) {
        for (OverlayDecider decider : getDeciders(screen)) {
            if (decider instanceof DisplayBoundsProvider) {
                Rectangle bounds = ((DisplayBoundsProvider<T>) decider).getScreenBounds(screen);
                
                if (bounds != null) {
                    return bounds;
                }
            }
        }
        return new Rectangle();
    }
    
    @Override
    public <T extends class_437> Rectangle getOverlayBounds(DisplayPanelLocation location, T screen) {
        class_1041 window = class_310.method_1551().method_22683();
        int scaledWidth = window.method_4486();
        int scaledHeight = window.method_4502();
        Rectangle screenBounds = getScreenBounds(screen);
        if (screenBounds.isEmpty()) return new Rectangle();
        if (location == DisplayPanelLocation.LEFT) {
            if (screenBounds.x < 10) return new Rectangle();
            return new Rectangle(2, 0, screenBounds.x - 2, scaledHeight);
        } else {
            if (scaledWidth - screenBounds.getMaxX() < 10) return new Rectangle();
            return new Rectangle(screenBounds.getMaxX() + 2, 0, scaledWidth - screenBounds.getMaxX() - 4, scaledHeight);
        }
    }
    
    @Nullable
    @Override
    public <T extends class_437> EntryStack<?> getFocusedStack(T screen, Point mouse) {
        for (FocusedStackProvider provider : focusedStackProviders) {
            CompoundEventResult<EntryStack<?>> result = Objects.requireNonNull(provider.provide(screen, mouse));
            if (result.isTrue()) {
                if (result != null && !result.object().isEmpty())
                    return result.object();
                return null;
            } else if (result.isFalse())
                return null;
        }
        
        return null;
    }
    
    @Override
    public void registerDecider(OverlayDecider decider) {
        deciders.add(decider);
        deciders.sort(Comparator.reverseOrder());
        cache.clear();
        tmpScreen.remove();
        InternalLogger.getInstance().debug("Added overlay decider: %s [%.2f priority]", decider, decider.getPriority());
    }
    
    @Override
    public void registerFocusedStack(FocusedStackProvider provider) {
        focusedStackProviders.add(provider);
        focusedStackProviders.sort(Comparator.reverseOrder());
        InternalLogger.getInstance().debug("Added focused stack provider: %s [%.2f priority]", provider, provider.getPriority());
    }
    
    @Override
    public <T extends class_437> void registerDraggableStackProvider(DraggableStackProvider<T> provider) {
        registerDraggableComponentProvider(provider);
    }
    
    @Override
    public <T extends class_437> void registerDraggableStackVisitor(DraggableStackVisitor<T> visitor) {
        registerDraggableComponentVisitor(visitor);
    }
    
    @Override
    public <T extends class_437, A> void registerDraggableComponentProvider(DraggableComponentProvider<T, A> provider) {
        draggableProviders.add((DraggableComponentProvider<class_437, Object>) provider);
        draggableProviders.sort(Comparator.reverseOrder());
        InternalLogger.getInstance().debug("Added draggable component provider: %s [%.2f priority]", provider, provider.getPriority());
    }
    
    @Override
    public <T extends class_437> void registerDraggableComponentVisitor(DraggableComponentVisitor<T> visitor) {
        draggableVisitors.add((DraggableComponentVisitor<class_437>) visitor);
        draggableVisitors.sort(Comparator.reverseOrder());
        InternalLogger.getInstance().debug("Added draggable component visitor: %s [%.2f priority]", visitor, visitor.getPriority());
    }
    
    @Override
    public Iterable<DraggableComponentProvider<class_437, Object>> getDraggableComponentProviders() {
        return Collections.unmodifiableList(draggableProviders);
    }
    
    @Override
    public Iterable<DraggableComponentVisitor<class_437>> getDraggableComponentVisitors() {
        return Collections.unmodifiableList(draggableVisitors);
    }
    
    @Override
    public ExclusionZones exclusionZones() {
        return exclusionZones;
    }
    
    @Override
    public <C extends class_1703, T extends class_465<C>> void registerContainerClickArea(SimpleClickArea<T> area, Class<? extends T> screenClass, CategoryIdentifier<?>... categories) {
        registerClickArea(screen -> {
            Rectangle rectangle = area.provide(screen).clone();
            rectangle.translate(screen.field_2776, screen.field_2800);
            return rectangle;
        }, screenClass, categories);
    }
    
    @Override
    public <T extends class_437> void registerClickArea(Class<? extends T> screenClass, ClickArea<T> area) {
        clickAreas.put(screenClass, area);
        InternalLogger.getInstance().debug("Added click area provider for %s: %s", screenClass.getName(), area);
    }
    
    @Override
    public <T extends class_437> List<ClickArea.Result> evaluateClickArea(Class<T> screenClass, ClickArea.ClickAreaContext<T> context) {
        List<ClickArea.Result> results = new ArrayList<>();
        for (ClickArea<?> area : this.clickAreas.get(screenClass)) {
            ClickArea.Result result = ((ClickArea<T>) area).handle(context);
            
            if (result.isSuccessful()) {
                results.add(result);
            }
        }
        return results;
    }
    
    @Override
    public void startReload() {
        clickAreas.clear();
        deciders.clear();
        cache.clear();
        focusedStackProviders.clear();
        draggableProviders.clear();
        draggableVisitors.clear();
        tmpScreen.remove();
        if (this.lastRendererProvider != null) {
            this.lastRendererProvider.onRemoved();
        }
        this.lastRendererProvider = null;
        
        registerDefault();
    }
    
    @Override
    public void endReload() {
        if (this.lastRendererProvider != null) {
            this.lastRendererProvider.onRemoved();
        }
        this.lastRendererProvider = null;
        InternalLogger.getInstance().debug("Registered %d overlay deciders and %d exclusion zones", deciders.size(), exclusionZones.getZonesCount());
    }
    
    private void registerDefault() {
        registerDecider(this.exclusionZones = new ExclusionZonesImpl());
        registerDecider(new DisplayBoundsProvider<class_465<?>>() {
            @Override
            public Rectangle getScreenBounds(class_465<?> screen) {
                return new Rectangle(screen.field_2776, screen.field_2800, screen.field_2792, screen.field_2779);
            }
            
            @Override
            public <R extends class_437> boolean isHandingScreen(Class<R> screen) {
                return class_465.class.isAssignableFrom(screen);
            }
            
            @Override
            public <R extends class_437> class_1269 shouldScreenBeOverlaid(R screen) {
                return screen instanceof class_465<?> ? class_1269.field_5812 : class_1269.field_5811;
            }
            
            @Override
            public double getPriority() {
                return -10.0;
            }
        });
        registerDecider(new DisplayBoundsProvider<AbstractDisplayViewingScreen>() {
            @Override
            public Rectangle getScreenBounds(AbstractDisplayViewingScreen screen) {
                return screen.getBounds();
            }
            
            @Override
            public <R extends class_437> boolean isHandingScreen(Class<R> screen) {
                return AbstractDisplayViewingScreen.class.isAssignableFrom(screen);
            }
            
            @Override
            public <R extends class_437> class_1269 shouldScreenBeOverlaid(R screen) {
                return class_1269.field_5812;
            }
            
            @Override
            public double getPriority() {
                return -10.0;
            }
        });
        registerDecider(new OverlayDecider() {
            @Override
            public <R extends class_437> boolean isHandingScreen(Class<R> screen) {
                return true;
            }
            
            @Override
            public OverlayRendererProvider getRendererProvider() {
                return DefaultScreenOverlayRenderer.INSTANCE;
            }
            
            @Override
            public double getPriority() {
                return -20.0;
            }
        });
        registerFocusedStack(new FocusedStackProvider() {
            @Override
            public CompoundEventResult<EntryStack<?>> provide(class_437 screen, Point mouse) {
                if (screen instanceof class_465<?> containerScreen) {
                    if (containerScreen.field_2787 != null && !containerScreen.field_2787.method_7677().method_7960())
                        return CompoundEventResult.interruptTrue(EntryStacks.of(containerScreen.field_2787.method_7677()));
                }
                return CompoundEventResult.pass();
            }
            
            @Override
            public double getPriority() {
                return -10.0;
            }
        });
        registerDraggableComponentProvider(DraggableComponentProviderWidget.from(context ->
                Widgets.walk(context.getScreen().method_25396(), DraggableComponentProviderWidget.class::isInstance)));
        registerDraggableComponentVisitor(DraggableComponentVisitorWidget.from(context ->
                Widgets.walk(context.getScreen().method_25396(), DraggableComponentVisitorWidget.class::isInstance)));
    }
    
    public OverlayRendererProvider getLastRendererProvider(class_437 screen) {
        for (OverlayDecider decider : getDeciders(screen)) {
            if (decider.isHandingScreen(screen.getClass())) {
                OverlayRendererProvider provider = decider.getRendererProvider();
                
                if (provider != null) {
                    return applyRendererProvider(provider);
                }
            }
        }
        
        return applyRendererProvider(null);
    }
    
    private OverlayRendererProvider applyRendererProvider(OverlayRendererProvider provider) {
        if (provider != this.lastRendererProvider) {
            if (this.lastRendererProvider != null) {
                this.lastRendererProvider.onRemoved();
            }
            this.lastRendererProvider = provider;
            if (this.lastRendererProvider != null) {
                this.lastRendererProvider.onApplied(new OverlayRendererProvider.Sink() {
                    @Override
                    public void render(class_332 graphics, int mouseX, int mouseY, float delta) {
                        if (ScreenRegistryImpl.this.lastRendererProvider == provider) {
                            ScreenOverlayImpl.getInstance().method_25394(graphics, mouseX, mouseY, delta);
                        } else {
                            InternalLogger.getInstance().warn("Renderer provider %s still tries to render after being removed!", provider);
                        }
                    }
                    
                    @Override
                    public void lateRender(class_332 graphics, int mouseX, int mouseY, float delta) {
                        if (ScreenRegistryImpl.this.lastRendererProvider == provider) {
                            ScreenOverlayImpl.getInstance().lateRender(graphics, mouseX, mouseY, delta);
                        } else {
                            InternalLogger.getInstance().warn("Renderer provider %s still tries to render after being removed!", provider);
                        }
                    }
                    
                    @Override
                    public ScreenOverlay getOverlay() {
                        return ScreenOverlayImpl.getInstance();
                    }
                });
            }
        }
        
        return provider;
    }
}
