/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.widget.search;

import com.google.common.collect.Lists;
import com.mojang.datafixers.util.Pair;
import me.shedaniel.clothconfig2.api.animator.NumberAnimator;
import me.shedaniel.clothconfig2.api.animator.ValueAnimator;
import me.shedaniel.math.Color;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.math.impl.PointHelper;
import me.shedaniel.rei.api.client.REIRuntime;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.gui.config.SearchFieldLocation;
import me.shedaniel.rei.api.client.gui.config.SyntaxHighlightingMode;
import me.shedaniel.rei.api.client.gui.widgets.Tooltip;
import me.shedaniel.rei.api.client.gui.widgets.Widgets;
import me.shedaniel.rei.api.common.util.CollectionUtils;
import me.shedaniel.rei.impl.client.REIRuntimeImpl;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.hints.HintProvider;
import me.shedaniel.rei.impl.client.gui.text.TextTransformations;
import me.shedaniel.rei.impl.client.gui.widget.basewidgets.TextFieldWidget;
import me.shedaniel.rei.impl.client.search.argument.type.ArgumentType;
import me.shedaniel.rei.impl.client.search.argument.type.ArgumentTypesRegistry;
import me.shedaniel.rei.impl.client.search.argument.type.TextArgumentType;
import net.minecraft.class_1074;
import net.minecraft.class_1109;
import net.minecraft.class_11905;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3417;
import net.minecraft.class_3545;
import net.minecraft.class_3675;
import net.minecraft.class_5251;
import net.minecraft.class_5481;
import org.jetbrains.annotations.ApiStatus;
import org.lwjgl.glfw.GLFW;

import java.util.List;
import java.util.Objects;
import java.util.OptionalDouble;
import java.util.function.Consumer;

@ApiStatus.Internal
public class OverlaySearchField extends TextFieldWidget implements TextFieldWidget.TextFormatter {
    public static boolean isHighlighting = false;
    private static final class_2583 SPLITTER_STYLE = class_2583.field_24360.method_10977(class_124.field_1080);
    private static final class_2583 QUOTES_STYLE = class_2583.field_24360.method_10977(class_124.field_1065);
    private static final class_2583 ERROR_STYLE = class_2583.field_24360.method_27703(class_5251.method_27717(0xff5555));
    private boolean previouslyClicking = false;
    private final OverlaySearchFieldSyntaxHighlighter highlighter = new OverlaySearchFieldSyntaxHighlighter(getText());
    public long keybindFocusTime = -1;
    public int keybindFocusKey = -1;
    public boolean isMain = true;
    protected class_3545<Long, Point> lastClickedDetails = null;
    private final List<String> history = Lists.newArrayListWithCapacity(100);
    private final NumberAnimator<Double> progress = ValueAnimator.ofDouble();
    
    public OverlaySearchField(int x, int y, int width, int height) {
        super(x, y, width, height);
        setMaxLength(10000);
        setFormatter(this);
        super.setResponder(highlighter);
    }
    
    @Override
    public class_5481 format(TextFieldWidget widget, String text, int index) {
        boolean isPlain = ConfigObject.getInstance().getSyntaxHighlightingMode() == SyntaxHighlightingMode.PLAIN || ConfigObject.getInstance().getSyntaxHighlightingMode() == SyntaxHighlightingMode.PLAIN_UNDERSCORED;
        boolean hasUnderscore = ConfigObject.getInstance().getSyntaxHighlightingMode() == SyntaxHighlightingMode.PLAIN_UNDERSCORED || ConfigObject.getInstance().getSyntaxHighlightingMode() == SyntaxHighlightingMode.COLORFUL_UNDERSCORED;
        return TextTransformations.forwardWithTransformation(text, (s, charIndex, c) -> {
            byte arg = highlighter.highlighted[charIndex + index];
            class_2583 style = class_2583.field_24360;
            if (isMain && ScreenOverlayImpl.getEntryListWidget().isEmpty() && !getText().isEmpty()) {
                style = ERROR_STYLE;
            }
            if (arg > 0) {
                ArgumentType<?, ?> argumentType = ArgumentTypesRegistry.ARGUMENT_TYPE_LIST.get((arg - 1) / 2);
                if (!isPlain) {
                    style = argumentType.getHighlightedStyle();
                }
                if (!(argumentType instanceof TextArgumentType) && hasUnderscore && arg % 2 == 1) {
                    style = style.method_30938(true);
                }
            } else if (!isPlain) {
                if (arg == -1) {
                    style = SPLITTER_STYLE;
                } else if (arg == -2) {
                    style = QUOTES_STYLE;
                }
            }
            
            if (containsMouse(PointHelper.ofMouse()) || method_25370()) {
                return style;
            }
            return style.method_27703(class_5251.method_27717(Color.ofOpaque(style.method_10973() == null ? -1 : style.method_10973().method_27716()).brighter(0.75).getColor()));
        });
    }
    
    @Override
    public void setResponder(Consumer<String> responder) {
        super.setResponder(highlighter.andThen(responder));
    }
    
    @Override
    public void method_25365(boolean focused) {
        if (method_25370() != focused && isMain)
            addToHistory(getText());
        super.method_25365(focused);
    }
    
    @ApiStatus.Internal
    public void addToHistory(String text) {
        if (!text.isEmpty()) {
            history.removeIf(str -> str.equalsIgnoreCase(text));
            if (history.size() > 100)
                history.remove(0);
            history.add(text);
        }
    }
    
    public void laterRender(class_332 graphics, int mouseX, int mouseY, float delta) {
        progress.update(delta);
        if (isMain) drawHint(graphics, mouseX, mouseY);
    }
    
    private void drawHint(class_332 graphics, int mouseX, int mouseY) {
        boolean mouseDown = GLFW.glfwGetMouseButton(class_310.method_1551().method_22683().method_4490(), GLFW.GLFW_MOUSE_BUTTON_LEFT) != 0;
        boolean clicking = false;
        if (mouseDown != previouslyClicking) {
            previouslyClicking = mouseDown;
            clicking = mouseDown;
        }
        
        List<HintProvider> hintProviders = REIRuntimeImpl.getInstance().getHintProviders();
        List<Pair<HintProvider, class_2561>> hints = CollectionUtils.flatMap(hintProviders, provider ->
                CollectionUtils.map(provider.provide(), component -> new Pair<>(provider, component)));
        if (hints.isEmpty()) return;
        int width = getBounds().getWidth() - 4;
        List<Pair<HintProvider, class_5481>> sequences = CollectionUtils.flatMap(hints, pair ->
                CollectionUtils.map(font.method_1728(pair.getSecond(), width - 6), sequence -> new Pair<>(pair.getFirst(), sequence)));
        OptionalDouble progress = hintProviders.stream().map(HintProvider::getProgress).filter(Objects::nonNull).mapToDouble(Double::doubleValue)
                .average();
        List<HintProvider.HintButton> buttons = hints.stream().map(Pair::getFirst).distinct()
                .map(HintProvider::getButtons)
                .flatMap(List::stream)
                .toList();
        boolean hasProgress = progress.isPresent();
        if (!hasProgress) {
            this.progress.setAs(0);
        } else {
            this.progress.setTo(progress.getAsDouble(), 200);
        }
        Color color = hints.stream()
                .map(Pair::getFirst)
                .distinct()
                .map(HintProvider::getColor)
                .reduce((color1, color2) -> {
                    int r = color1.getRed() - (color1.getRed() - color2.getRed()) / 2;
                    int g = color1.getGreen() - (color1.getGreen() - color2.getGreen()) / 2;
                    int b = color1.getBlue() - (color1.getBlue() - color2.getBlue()) / 2;
                    return Color.ofRGBA(r, g, b, (color1.getAlpha() + color2.getAlpha()) / 2);
                }).orElse(Color.ofTransparent(0x50000000));
        int height = 6 + font.field_2000 * sequences.size() + (hasProgress ? 2 : 0) + (buttons.isEmpty() ? 0 : (int) Math.ceil(buttons.size() / 3.0) * 20);
        boolean top = ConfigObject.getInstance().getSearchFieldLocation() == SearchFieldLocation.TOP_SIDE;
        int x = getBounds().getX() + 2;
        int y = getBounds().getY() + (top ? getBounds().getHeight() : -height);
        if (new Rectangle(x - 1, y - 1, width + 2, height + 2).contains(mouseX, mouseY))
            ScreenOverlayImpl.getInstance().clearTooltips();
        int background = 0xf0100010;
        int color1 = color.getColor();
        int color2 = color.darker(2).getColor();
        if (!top) graphics.method_25296(x, y - 1, x + width, y, background, background);
        if (top)
            graphics.method_25296(x, y + height, x + width, y + height + 1, background, background);
        graphics.method_25296(x, y, x + width, y + height, background, background);
        graphics.method_25296(x - 1, y, x, y + height, background, background);
        graphics.method_25296(x + width, y, x + width + 1, y + height, background, background);
        graphics.method_25296(x, y + 1, x + 1, y + height - 1, color1, color2);
        graphics.method_25296(x + width - 1, y + 1, x + width, y + height - 1, color1, color2);
        if (!top) graphics.method_25296(x, y, x + width, y + 1, color1, color1);
        if (top) graphics.method_25296(x, y + height - 1, x + width, y + height, color2, color2);
        
        if (hasProgress) {
            int progressWidth = (int) Math.round(width * this.progress.doubleValue());
            graphics.method_25296(x + 1, y + height - 3, x + progressWidth - 1, y + height - 1, 0xffffffff, 0xffffffff);
        }
        
        for (int i = 0; i < sequences.size(); i++) {
            Pair<HintProvider, class_5481> pair = sequences.get(i);
            graphics.method_35720(font, pair.getSecond(), x + 3, y + 3 + font.field_2000 * i, -1);
            int lineWidth = font.method_30880(pair.getSecond());
            if (new Rectangle(x + 3, y + 3 + font.field_2000 * i, lineWidth, font.field_2000).contains(mouseX, mouseY)) {
                Tooltip tooltip = pair.getFirst().provideTooltip(new Point(mouseX, mouseY));
                if (tooltip != null) {
                    ScreenOverlayImpl.getInstance().clearTooltips();
                    ScreenOverlayImpl.getInstance().renderTooltip(graphics, tooltip);
                }
            }
        }
        
        int split = 2;
        for (HintProvider.HintButton button : buttons) {
            int x1 = x + 4 + ((width - 8 - 8) / split) * (buttons.indexOf(button) % split);
            int y1 = y + height - 20 - 20 * (int) Math.floor(buttons.indexOf(button) / (float) split);
            int x2 = x1 + (width - 8 - 8) / split;
            int y2 = y1 + 16;
            Rectangle bounds = new Rectangle(x1, y1, x2 - x1 - 4, y2 - y1);
            int buttonColor = bounds.contains(mouseX, mouseY) ? 0x8f8f8f8f : 0x66666666;
            graphics.method_25296(x1, y1, x2 - 4, y2, buttonColor, buttonColor);
            graphics.method_27535(font, button.name(), (x1 + x2 - 4 - font.method_27525(button.name())) / 2, y1 + 4, -1);
            
            if (bounds.contains(mouseX, mouseY) && clicking) {
                Widgets.produceClickSound();
                button.action().accept(bounds);
            }
        }
    }
    
    @Override
    protected void renderSuggestion(class_332 graphics, int x, int y) {
        int color;
        if (containsMouse(PointHelper.ofMouse()) || method_25370()) {
            color = 0xddeaeaea;
        } else {
            color = -6250336;
        }
        graphics.method_25303(this.font, this.font.method_27523(this.getSuggestion(), this.getWidth()), x, y, color);
    }
    
    @Override
    public void renderBorder(class_332 graphics) {
        isHighlighting = isHighlighting && ConfigObject.getInstance().isInventoryHighlightingAllowed();
        int borderColor;
        if (isMain && isHighlighting) {
            borderColor = 0xfff2ff0c;
        } else if (isMain && ScreenOverlayImpl.getEntryListWidget().isEmpty() && !getText().isEmpty()) {
            borderColor = 0xffff5555;
        } else {
            super.renderBorder(graphics);
            return;
        }
        graphics.method_25294(this.getBounds().x - 1, this.getBounds().y - 1, this.getBounds().x + this.getBounds().width + 1, this.getBounds().y + this.getBounds().height + 1, 0xff000000);
        graphics.method_25294(this.getBounds().x, this.getBounds().y, this.getBounds().x + this.getBounds().width, this.getBounds().y + this.getBounds().height, borderColor);
        graphics.method_25294(this.getBounds().x + 1, this.getBounds().y + 1, this.getBounds().x + this.getBounds().width - 1, this.getBounds().y + this.getBounds().height - 1, 0xff000000);
    }
    
    public int getManhattanDistance(Point point1, Point point2) {
        int e = Math.abs(point1.getX() - point2.getX());
        int f = Math.abs(point1.getY() - point2.getY());
        return e + f;
    }
    
    @Override
    public boolean method_25402(class_11909 event, boolean doubleClick) {
        boolean contains = containsMouse(event.comp_4798(), event.comp_4799());
        if (isVisible() && contains && event.method_74245() == 1)
            setText("");
        if (contains && event.method_74245() == 0 && isMain && ConfigObject.getInstance().isInventoryHighlightingAllowed())
            if (lastClickedDetails == null)
                lastClickedDetails = new class_3545<>(System.currentTimeMillis(), new Point(event.comp_4798(), event.comp_4799()));
            else if (System.currentTimeMillis() - lastClickedDetails.method_15442() > 1500)
                lastClickedDetails = null;
            else if (getManhattanDistance(lastClickedDetails.method_15441(), new Point(event.comp_4798(), event.comp_4799())) <= 25) {
                lastClickedDetails = null;
                isHighlighting = !isHighlighting;
                minecraft.method_1483().method_4873(class_1109.method_47978(class_3417.field_15015, 1.0F));
            } else {
                lastClickedDetails = new class_3545<>(System.currentTimeMillis(), new Point(event.comp_4798(), event.comp_4799()));
            }
        return super.method_25402(event, doubleClick);
    }
    
    @Override
    public boolean method_25404(class_11908 event) {
        if (this.isVisible() && this.method_25370() && isMain)
            if (event.comp_4795() == 257 || event.comp_4795() == 335) {
                addToHistory(getText());
                method_25365(false);
                return true;
            } else if (event.comp_4795() == 265) {
                int i = history.indexOf(getText()) - 1;
                if (i < -1 && getText().isEmpty())
                    i = history.size() - 1;
                else if (i < -1) {
                    addToHistory(getText());
                    i = history.size() - 2;
                }
                if (i >= 0) {
                    setText(history.get(i));
                    return true;
                }
            } else if (event.comp_4795() == 264) {
                int i = history.indexOf(getText()) + 1;
                if (i > 0) {
                    setText(i < history.size() ? history.get(i) : "");
                    return true;
                }
            }
        return super.method_25404(event);
    }
    
    @Override
    public boolean method_16803(class_11908 event) {
        if (this.isVisible() && this.method_25370() && isMain && keybindFocusKey != -1) {
            keybindFocusTime = -1;
            keybindFocusKey = -1;
            return true;
        }
        return super.method_16803(event);
    }
    
    @Override
    public boolean method_25400(class_11905 event) {
        if (isMain && System.currentTimeMillis() - keybindFocusTime < 1000 && keybindFocusKey != -1 && class_3675.method_15987(class_310.method_1551().method_22683(), keybindFocusKey)) {
            keybindFocusTime = -1;
            keybindFocusKey = -1;
            return true;
        }
        return super.method_25400(event);
    }
    
    @Override
    public boolean containsMouse(double mouseX, double mouseY) {
        return (!isMain || REIRuntime.getInstance().getOverlay().get().isNotInExclusionZones(mouseX, mouseY)) && super.containsMouse(mouseX, mouseY);
    }
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        setSuggestion(!method_25370() && getText().isEmpty() ? class_1074.method_4662("text.rei.search.field.suggestion") : null);
        super.method_25394(graphics, mouseX, mouseY, delta);
    }
}
