/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.widget;

import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.favorites.FavoriteEntry;
import me.shedaniel.rei.api.client.favorites.FavoriteEntryType;
import me.shedaniel.rei.api.client.gui.drag.DraggedAcceptorResult;
import me.shedaniel.rei.api.client.gui.drag.DraggingContext;
import me.shedaniel.rei.api.client.gui.drag.component.DraggableComponent;
import me.shedaniel.rei.api.client.gui.drag.component.DraggableComponentProviderWidget;
import me.shedaniel.rei.api.client.gui.widgets.*;
import me.shedaniel.rei.api.client.gui.widgets.utils.PanelTextures;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.util.CollectionUtils;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.widget.favorites.FavoritesListWidget;
import me.shedaniel.rei.impl.display.DisplaySpec;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_332;
import net.minecraft.class_437;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Objects;
import java.util.stream.StreamSupport;

public class DisplayCompositeWidget extends DelegateWidgetWithBounds implements DraggableComponentProviderWidget<Object> {
    private final DisplaySpec display;
    private final List<Widget> widgets;
    
    public DisplayCompositeWidget(DisplaySpec display, List<Widget> widgets, Rectangle bounds) {
        super(Widgets.concat(widgets), () -> bounds);
        this.display = display;
        this.widgets = widgets;
    }
    
    @Override
    @Nullable
    public DraggableComponent<Object> getHovered(DraggingContext<class_437> context, double mouseX, double mouseY) {
        return StreamSupport.stream(Widgets.<DraggableComponentProviderWidget<Object>>walk(widget.method_25396(), widget -> widget instanceof DraggableComponentProviderWidget).spliterator(), false)
                .map(widget -> widget.getHovered(context, mouseX, mouseY))
                .filter(Objects::nonNull)
                .findFirst()
                .orElseGet(() -> {
                    if (containsMouse(mouseX, mouseY)) {
                        return (DraggableComponent<Object>) (DraggableComponent<?>) new DisplayDraggableComponent(
                                Widgets.concat(CollectionUtils.filterToList(widgets, w -> !(w instanceof Panel))),
                                display.provideInternalDisplay(), getBounds(), getBounds());
                    } else {
                        return null;
                    }
                });
    }
    
    @Override
    public boolean method_25404(class_11908 event) {
        if (super.method_25404(event)) {
            return true;
        }
        
        if (ConfigObject.getInstance().isFavoritesEnabled() && containsMouse(mouse())) {
            if (ConfigObject.getInstance().getFavoriteKeyCode().matchesKey(event.comp_4795(), event.comp_4796())) {
                FavoriteEntry favoriteEntry = FavoriteEntryType.registry().get(FavoriteEntryType.DISPLAY)
                        .fromArgs(display.provideInternalDisplay())
                        .result()
                        .orElse(null);
                if (favoriteEntry != null) {
                    ConfigObject.getInstance().getFavoriteEntries().add(favoriteEntry);
                    return true;
                }
            }
        }
        
        return false;
    }
    
    @Override
    public boolean method_25406(class_11909 event) {
        if (super.method_25406(event)) {
            return true;
        }
        
        if (ConfigObject.getInstance().isFavoritesEnabled() && containsMouse(event.comp_4798(), event.comp_4799())) {
            if (ConfigObject.getInstance().getFavoriteKeyCode().matchesMouse(event.method_74245())) {
                FavoriteEntry favoriteEntry = FavoriteEntryType.registry().get(FavoriteEntryType.DISPLAY)
                        .fromArgs(display.provideInternalDisplay())
                        .result()
                        .orElse(null);
                if (favoriteEntry != null) {
                    ConfigObject.getInstance().getFavoriteEntries().add(favoriteEntry);
                    return true;
                }
            }
        }
        
        return false;
    }
    
    public static class DisplayDraggableComponent implements DraggableComponent<Display> {
        private final Widget widget;
        private final Display display;
        private final Rectangle originBounds;
        private final Rectangle bounds;
        private final Panel panel;
        private final Slot slot;
        public boolean onFavoritesRegion;
        
        public DisplayDraggableComponent(Widget widget, Display display, Rectangle originBounds, Rectangle bounds) {
            this.widget = widget;
            this.display = display;
            this.originBounds = originBounds;
            this.bounds = bounds;
            this.panel = Widgets.createRecipeBase(bounds.clone());
            this.slot = Widgets.createSlot(new Rectangle())
                    .disableBackground()
                    .disableHighlight()
                    .disableTooltips();
            for (EntryIngredient ingredient : display.getOutputEntries()) {
                slot.entries(ingredient);
            }
        }
        
        @Override
        public int getWidth() {
            if (this.onFavoritesRegion) {
                return 18;
            }
            
            return bounds.width / 2;
        }
        
        @Override
        public int getHeight() {
            if (this.onFavoritesRegion) {
                return 18;
            }
            
            return bounds.height / 2;
        }
        
        @Override
        public Display get() {
            return display;
        }
        
        @Override
        public void render(class_332 graphics, Rectangle bounds, int mouseX, int mouseY, float delta) {
            if (DraggingContext.getInstance().isDraggingComponent()) {
                FavoritesListWidget favorites = ScreenOverlayImpl.getFavoritesListWidget();
                if (favorites != null) {
                    Rectangle favoritesBounds = favorites.getRegion().getBounds();
                    if (!this.onFavoritesRegion && new Rectangle(favoritesBounds.x + 5, favoritesBounds.y + 5, favoritesBounds.width - 10, favoritesBounds.height - 10)
                            .contains(DraggingContext.getInstance().getCurrentPosition())) {
                        this.onFavoritesRegion = true;
                    } else if (this.onFavoritesRegion && !favoritesBounds.contains(DraggingContext.getInstance().getCurrentPosition())) {
                        this.onFavoritesRegion = false;
                    }
                } else {
                    this.onFavoritesRegion = false;
                }
            } else {
                this.onFavoritesRegion = false;
            }
            
            graphics.method_51448().pushMatrix();
            if (bounds.width <= Math.max(18, this.bounds.width / 2 - 6) && bounds.height <= Math.max(18, this.bounds.height / 2 - 6) && this.onFavoritesRegion) {
                this.panel.texture(PanelTextures.LIGHTER);
                this.panel.getBounds().setBounds(bounds);
                this.panel.method_25394(graphics, mouseX, mouseY, delta);
                graphics.method_51448().pushMatrix();
                graphics.method_51448().translate(0, 0.5f);
                this.slot.getBounds().setBounds(bounds.getCenterX() - 7, bounds.getCenterY() - 7, 14, 14);
                this.slot.method_25394(graphics, mouseX, mouseY, delta);
                graphics.method_51448().popMatrix();
            } else {
                this.panel.texture(ConfigObject.getInstance().getRecipeBorderType());
                graphics.method_51448().pushMatrix();
                graphics.method_51448().translate(bounds.getX(), bounds.getY());
                graphics.method_51448().scale(bounds.width / (float) this.bounds.getWidth(), bounds.height / (float) this.bounds.getHeight());
                graphics.method_51448().translate(-this.bounds.getX(), -this.bounds.getY());
                this.panel.getBounds().setBounds(this.bounds);
                this.panel.method_25394(graphics, mouseX, mouseY, delta);
                graphics.method_51448().popMatrix();
                graphics.method_51448().translate(bounds.getX(), bounds.getY());
                graphics.method_51448().scale(bounds.width / (float) this.bounds.getWidth(), bounds.height / (float) this.bounds.getHeight());
                graphics.method_51448().translate(-this.bounds.getX(), -this.bounds.getY());
                widget.method_25394(graphics, -1000, -1000, delta);
            }
            graphics.method_51448().popMatrix();
        }
        
        @Override
        public void release(DraggedAcceptorResult result) {
            if (result == DraggedAcceptorResult.PASS) {
                DraggingContext.getInstance().renderBack(this, DraggingContext.getInstance().getCurrentBounds(), () -> originBounds);
            }
        }
        
        @Override
        public Rectangle getOriginBounds(Point mouse) {
            return originBounds.clone();
        }
    }
}
