/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.screen;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import it.unimi.dsi.fastutil.Pair;
import me.shedaniel.clothconfig2.api.ModifierKeyCode;
import me.shedaniel.clothconfig2.api.animator.ValueAnimator;
import me.shedaniel.math.Color;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.math.impl.PointHelper;
import me.shedaniel.rei.api.client.REIRuntime;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.gui.config.SearchFieldLocation;
import me.shedaniel.rei.api.client.gui.widgets.Button;
import me.shedaniel.rei.api.client.gui.widgets.Panel;
import me.shedaniel.rei.api.client.gui.widgets.Widget;
import me.shedaniel.rei.api.client.gui.widgets.Widgets;
import me.shedaniel.rei.api.client.registry.category.ButtonArea;
import me.shedaniel.rei.api.client.registry.category.CategoryRegistry;
import me.shedaniel.rei.api.client.registry.display.DisplayCategory;
import me.shedaniel.rei.api.client.registry.display.DisplayRegistry;
import me.shedaniel.rei.api.client.view.ViewSearchBuilder;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.util.CollectionUtils;
import me.shedaniel.rei.impl.client.REIRuntimeImpl;
import me.shedaniel.rei.impl.client.gui.InternalTextures;
import me.shedaniel.rei.impl.client.gui.RecipeDisplayExporter;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.toast.ExportRecipeIdentifierToast;
import me.shedaniel.rei.impl.client.gui.widget.*;
import me.shedaniel.rei.impl.client.gui.widget.basewidgets.PanelWidget;
import me.shedaniel.rei.impl.display.DisplaySpec;
import net.minecraft.class_1074;
import net.minecraft.class_10799;
import net.minecraft.class_11905;
import net.minecraft.class_11908;
import net.minecraft.class_11909;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_364;
import net.minecraft.class_410;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

@ApiStatus.Internal
public class DefaultDisplayViewingScreen extends AbstractDisplayViewingScreen {
    private static final int INNER_PADDING_Y = 36;
    private static final int OUTER_PADDING_TOP = 2;
    private static final int OUTER_PADDING_BOTTOM = 2;
    private static final int DISPLAY_GAP = 4;
    private final Map<Rectangle, Pair<DisplaySpec, List<Widget>>> recipeBounds = Maps.newHashMap();
    private List<Widget> widgets = Lists.newArrayList();
    public int page;
    @Nullable
    private Panel workingStationsBaseWidget;
    private Button recipeBack, recipeNext, categoryBack, categoryNext;
    private final int bestWidthDisplay;
    
    public DefaultDisplayViewingScreen(Map<DisplayCategory<?>, List<DisplaySpec>> categoriesMap, @Nullable CategoryIdentifier<?> category) {
        super(categoriesMap, category);
        this.bounds = new Rectangle(0, 0, 176, 150);
        //noinspection RedundantCast
        List<Integer> list = CollectionUtils.mapAndFilter(categoriesMap.entrySet(), Objects::nonNull, entry -> ((Optional<Integer>) CollectionUtils.<DisplaySpec, Integer>mapAndMax(entry.getValue(),
                display -> ((DisplayCategory<Display>) entry.getKey()).getDisplayWidth(display.provideInternalDisplay()), Comparator.naturalOrder())).orElse(null));
        list.sort(Comparator.naturalOrder());
        int mode = list.stream().collect(Collectors.groupingBy(Function.identity(), Collectors.counting())).entrySet().stream()
                .max(Map.Entry.comparingByValue())
                .map(Map.Entry::getKey)
                .orElse(150);
        int median = list.size() % 2 == 0 ? (list.get(list.size() / 2) + list.get(list.size() / 2 - 1)) / 2 : list.get(list.size() / 2);
        this.bestWidthDisplay = (int) Math.round((mode * 0.5 + median * 1.5) / 2.0);
    }
    
    @Override
    public void recalculateCategoryPage() {
        this.categoryPages = -1;
    }
    
    @Nullable
    public Panel getWorkingStationsBaseWidget() {
        return workingStationsBaseWidget;
    }
    
    @Override
    public boolean method_25404(class_11908 event) {
        if (ConfigObject.getInstance().getNextPageKeybind().matchesKey(event.comp_4795(), event.comp_4796())) {
            if (recipeNext.isEnabled())
                recipeNext.onClick();
            return recipeNext.isEnabled();
        } else if (ConfigObject.getInstance().getPreviousPageKeybind().matchesKey(event.comp_4795(), event.comp_4796())) {
            if (recipeBack.isEnabled())
                recipeBack.onClick();
            return recipeBack.isEnabled();
        }
        for (class_364 element : method_25396())
            if (element.method_25404(event))
                return true;
        if (event.method_74231()) {
            class_310.method_1551().method_1507(REIRuntime.getInstance().getPreviousScreen());
            return true;
        }
        return super.method_25404(event);
    }
    
    @Override
    public void method_25426() {
        super.method_25426();
        this.method_25396().clear();
        this.recipeBounds.clear();
        this.widgets.clear();
//        int maxWidthDisplay = CollectionUtils.<DisplaySpec, Integer>mapAndMax(getCurrentDisplayed(), display -> getCurrentCategory().getDisplayWidth(display.provideInternalDisplay()), Comparator.naturalOrder()).orElse(150);
//        int guiWidth = Math.max(maxWidthDisplay + 10 + 14 + 14, 190);
        int guiWidth = Math.max(bestWidthDisplay + 10 + 14 + 14, 190);
        this.tabs.initTabsSize(guiWidth);
        
        int topMargin = OUTER_PADDING_TOP + this.tabs.tabSize() - 2 + (categories.size() > this.tabs.tabsPerPage() ? 16 : 0);
        int bottomMargin = OUTER_PADDING_BOTTOM + (ConfigObject.getInstance().getSearchFieldLocation() == SearchFieldLocation.CENTER ? 22 : 0);
        int largestHeight = Math.min(Math.max(field_22790 - topMargin - bottomMargin, 100), ConfigObject.getInstance().getMaxRecipesPageHeight());
        int maxHeight = Math.min(largestHeight, CollectionUtils.<DisplayCategory<?>, Integer>mapAndMax(categories,
                category -> INNER_PADDING_Y + (category.getDisplayHeight() + DISPLAY_GAP) * Math.max(1, Math.min(getRecipesPerPage(largestHeight, category) + 1, Math.max(categoryMap.get(category).size(), ConfigObject.getInstance().getMaxRecipePerPage()))), Comparator.naturalOrder()).orElse(66));
        this.bounds = new Rectangle(field_22789 / 2 - guiWidth / 2, topMargin + (field_22790 - topMargin - bottomMargin) / 2 - maxHeight / 2, guiWidth, maxHeight);
        
        this.initTabs(guiWidth);
        this.widgets.addAll(this.tabs.widgets());
        
        this.page = class_3532.method_15340(page, 0, getCurrentTotalPages() - 1);
        this.widgets.add(categoryBack = Widgets.createButton(new Rectangle(bounds.getCenterX() - guiWidth / 2 + 5, bounds.getY() + 5, 12, 12), class_2561.method_43473())
                .onClick(button -> previousCategory()).tooltipLine(class_2561.method_43471("text.rei.previous_category")));
        this.widgets.add(Widgets.createClickableLabel(new Point(bounds.getCenterX(), bounds.getY() + 7), getCurrentCategory().getTitle(), clickableLabelWidget -> {
            ViewSearchBuilder.builder().addAllCategories().open();
        }).tooltip(class_2561.method_43471("text.rei.view_all_categories"), class_2561.method_43469("text.rei.view_all_categories.tooltip", CategoryRegistry.getInstance().stream().filter(config -> !DisplayRegistry.getInstance().get(config.getCategoryIdentifier()).isEmpty()).count()).method_27692(class_124.field_1063)));
        this.widgets.add(categoryNext = Widgets.createButton(new Rectangle(bounds.getCenterX() + guiWidth / 2 - 17, bounds.getY() + 5, 12, 12), class_2561.method_43470(""))
                .onClick(button -> nextCategory()).tooltipLine(class_2561.method_43471("text.rei.next_category")));
        this.categoryBack.setEnabled(categories.size() > 1);
        this.categoryNext.setEnabled(categories.size() > 1);
        
        this.widgets.add(recipeBack = Widgets.createButton(new Rectangle(bounds.getCenterX() - guiWidth / 2 + 5, bounds.getY() + 19, 12, 12), class_2561.method_43470(""))
                .onClick(button -> {
                    page--;
                    if (page < 0)
                        page = getCurrentTotalPages() - 1;
                    DefaultDisplayViewingScreen.this.method_25426();
                }).tooltipLine(class_2561.method_43471("text.rei.previous_page")));
        this.widgets.add(Widgets.createClickableLabel(new Point(bounds.getCenterX(), bounds.getY() + 21), class_2561.method_43473(), label -> {
            if (!class_310.method_1551().method_74187()) {
                page = 0;
                DefaultDisplayViewingScreen.this.method_25426();
            } else {
                ScreenOverlayImpl.getInstance().choosePageWidget = new DefaultDisplayChoosePageWidget(page -> {
                    DefaultDisplayViewingScreen.this.page = page;
                    DefaultDisplayViewingScreen.this.method_25426();
                }, page, getCurrentTotalPages());
            }
        }).onRender((matrices, label) -> {
            label.setMessage(class_2561.method_43470(String.format("%d/%d", page + 1, getCurrentTotalPages())));
            label.setClickable(getCurrentTotalPages() > 1);
        }).tooltipFunction(label -> label.isClickable() ? new class_2561[]{class_2561.method_43471("text.rei.go_back_first_page"), class_2561.method_43470(" "), class_2561.method_43469("text.rei.shift_click_to", class_2561.method_43471("text.rei.choose_page")).method_27692(class_124.field_1080)} : null));
        this.widgets.add(recipeNext = Widgets.createButton(new Rectangle(bounds.getCenterX() + guiWidth / 2 - 17, bounds.getY() + 19, 12, 12), class_2561.method_43470(""))
                .onClick(button -> {
                    page++;
                    if (page >= getCurrentTotalPages())
                        page = 0;
                    DefaultDisplayViewingScreen.this.method_25426();
                }).tooltipLine(class_2561.method_43471("text.rei.next_page")));
        
        this.widgets.add(Widgets.createDrawableWidget((graphics, mouseX, mouseY, delta) -> {
            Rectangle recipeBackBounds = recipeBack.getBounds();
            Rectangle recipeNextBounds = recipeNext.getBounds();
            Rectangle categoryBackBounds = categoryBack.getBounds();
            Rectangle categoryNextBounds = categoryNext.getBounds();
            graphics.method_51448().pushMatrix();
            graphics.method_51448().translate(0.5f, 0.5f);
            graphics.method_25290(class_10799.field_56883, InternalTextures.ARROW_LEFT_TEXTURE, recipeBackBounds.x + 2, recipeBackBounds.y + 2, 0, 0, 8, 8, 8, 8);
            graphics.method_25290(class_10799.field_56883, InternalTextures.ARROW_LEFT_TEXTURE, categoryBackBounds.x + 2, categoryBackBounds.y + 2, 0, 0, 8, 8, 8, 8);
            graphics.method_51448().translate(-0.5f, 0);
            graphics.method_25290(class_10799.field_56883, InternalTextures.ARROW_RIGHT_TEXTURE, recipeNextBounds.x + 2, recipeNextBounds.y + 2, 0, 0, 8, 8, 8, 8);
            graphics.method_25290(class_10799.field_56883, InternalTextures.ARROW_RIGHT_TEXTURE, categoryNextBounds.x + 2, categoryNextBounds.y + 2, 0, 0, 8, 8, 8, 8);
            graphics.method_51448().popMatrix();
        }));
        
        this.recipeBack.setEnabled(getCurrentTotalPages() > 1);
        this.recipeNext.setEnabled(getCurrentTotalPages() > 1);
        initDisplays();
        this.widgets.add(0, new PanelWidget(bounds));
        this.widgets.add(1, Widgets.createDrawableWidget((graphics, mouseX, mouseY, delta) -> {
            graphics.method_25294(bounds.getCenterX() - guiWidth / 2 + 17, bounds.y + 5, bounds.getCenterX() + guiWidth / 2 - 17, bounds.y + 17, darkStripesColor.value().getColor());
            graphics.method_25294(bounds.getCenterX() - guiWidth / 2 + 17, bounds.y + 19, bounds.getCenterX() + guiWidth / 2 - 17, bounds.y + 31, darkStripesColor.value().getColor());
        }));
        initWorkstations(widgets);
        
        method_25396().addAll(widgets);
    }
    
    private void initDisplays() {
        Optional<ButtonArea> plusButtonArea = CategoryRegistry.getInstance().get(getCurrentCategoryId()).getPlusButtonArea();
        int displayHeight = getCurrentCategory().getDisplayHeight();
        List<DisplaySpec> currentDisplayed = getCurrentDisplayed();
        for (int i = 0; i < currentDisplayed.size(); i++) {
            final DisplaySpec display = currentDisplayed.get(i);
            final Supplier<Display> displaySupplier = display::provideInternalDisplay;
            int displayWidth = getCurrentCategory().getDisplayWidth(displaySupplier.get());
            final Rectangle displayBounds = new Rectangle(getBounds().getCenterX() - displayWidth / 2, getBounds().getCenterY() + 16 - displayHeight * (getRecipesPerPage() + 1) / 2 - 2 * (getRecipesPerPage() + 1) + displayHeight * i + DISPLAY_GAP * i, displayWidth, displayHeight);
            List<Widget> setupDisplay;
            try {
                setupDisplay = getCurrentCategoryView(display.provideInternalDisplay()).setupDisplay(display.provideInternalDisplay(), displayBounds);
            } catch (Throwable throwable) {
                throwable.printStackTrace();
                setupDisplay = new ArrayList<>();
                setupDisplay.add(Widgets.createRecipeBase(displayBounds).color(0xFFBB0000));
                setupDisplay.add(Widgets.createLabel(new Point(displayBounds.getCenterX(), displayBounds.getCenterY() - 8), class_2561.method_43470("Failed to initiate setupDisplay")));
                setupDisplay.add(Widgets.createLabel(new Point(displayBounds.getCenterX(), displayBounds.getCenterY() + 1), class_2561.method_43470("Check console for error")));
            }
            setupTags(setupDisplay);
            transformFiltering(setupDisplay);
            transformIngredientNotice(setupDisplay, ingredientStackToNotice);
            transformResultNotice(setupDisplay, resultStackToNotice);
            unifyIngredients(setupDisplay);
            for (EntryWidget widget : Widgets.<EntryWidget>walk(widgets, EntryWidget.class::isInstance)) {
                widget.removeTagMatch = true;
            }
            this.recipeBounds.put(displayBounds, Pair.of(display, setupDisplay));
            this.widgets.add(new DisplayCompositeWidget(display, setupDisplay, displayBounds));
            if (plusButtonArea.isPresent()) {
                this.widgets.add(InternalWidgets.createAutoCraftingButtonWidget(displayBounds, plusButtonArea.get().get(displayBounds), class_2561.method_43470(plusButtonArea.get().getButtonText()), displaySupplier, display::provideInternalDisplayIds, setupDisplay, getCurrentCategory()));
            }
        }
    }
    
    private void initWorkstations(List<Widget> widgets) {
        workingStationsBaseWidget = null;
        List<EntryIngredient> workstations = CategoryRegistry.getInstance().get(getCurrentCategoryId()).getWorkstations();
        if (!workstations.isEmpty()) {
            int hh = class_3532.method_15375((bounds.height - 16) / 18f);
            int actualHeight = Math.min(hh, workstations.size());
            int innerWidth = class_3532.method_15386(workstations.size() / ((float) hh));
            int xx = bounds.x - (8 + innerWidth * 16) + 6;
            int yy = bounds.y + 16;
            List<Widget> toAdd = new ArrayList<>();
            toAdd.add(workingStationsBaseWidget = Widgets.createCategoryBase(new Rectangle(xx - 5, yy - 5, 15 + innerWidth * 16, 10 + actualHeight * 16)));
            toAdd.add(Widgets.createSlotBase(new Rectangle(xx - 1, yy - 1, innerWidth * 16 + 2, actualHeight * 16 + 2)));
            int index = 0;
            xx += (innerWidth - 1) * 16;
            for (EntryIngredient workingStation : workstations) {
                toAdd.add(new WorkstationSlotWidget(xx, yy, workingStation));
                index++;
                yy += 16;
                if (index >= hh) {
                    index = 0;
                    yy = bounds.y + 16;
                    xx -= 16;
                }
            }
            widgets.addAll(0, toAdd);
        }
    }
    
    public List<Widget> widgets() {
        widgets.sort(Comparator.comparingDouble(Widget::getZRenderingPriority));
        return widgets;
    }
    
    public List<DisplaySpec> getCurrentDisplayed() {
        List<DisplaySpec> list = Lists.newArrayList();
        int recipesPerPage = getRecipesPerPage();
        List<DisplaySpec> displays = categoryMap.get(getCurrentCategory());
        for (int i = 0; i <= recipesPerPage; i++) {
            if (page * (recipesPerPage + 1) + i < displays.size()) {
                list.add(displays.get(page * (recipesPerPage + 1) + i));
            }
        }
        return list;
    }
    
    public int getPage() {
        return page;
    }
    
    public int getCategoryPage() {
        return categoryPages;
    }
    
    private int getRecipesPerPage() {
        return getRecipesPerPage(this.bounds.height, getCurrentCategory());
    }
    
    private static int getRecipesPerPage(int totalHeight, DisplayCategory<?> category) {
        if (category.getFixedDisplaysPerPage() > 0)
            return category.getFixedDisplaysPerPage() - 1;
        int height = category.getDisplayHeight();
        return class_3532.method_15340(class_3532.method_15357(((double) totalHeight - INNER_PADDING_Y) / ((double) height + DISPLAY_GAP)) - 1, 0, Math.min(ConfigObject.getInstance().getMaxRecipePerPage() - 1, category.getMaximumDisplaysPerPage() - 1));
    }
    
    private final ValueAnimator<Color> darkStripesColor = ValueAnimator.ofColor()
            .withConvention(() -> Color.ofTransparent(REIRuntime.getInstance().isDarkThemeEnabled() ? 0xFF404040 : 0xFF9E9E9E), ValueAnimator.typicalTransitionTime());
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        darkStripesColor.update(delta);
        super.method_25394(graphics, mouseX, mouseY, delta);
        getOverlay().method_25394(graphics, mouseX, mouseY, delta);
        for (Widget widget : widgets()) {
            widget.method_25394(graphics, mouseX, mouseY, delta);
        }
        {
            ModifierKeyCode export = ConfigObject.getInstance().getExportImageKeybind();
            if (export.matchesCurrentKey() || export.matchesCurrentMouse()) {
                for (Rectangle bounds : Iterables.concat(recipeBounds.keySet(), Iterables.transform(getTabs(), TabWidget::getBounds))) {
                    if (bounds.contains(mouseX, mouseY)) {
                        graphics.method_25296(bounds.x, bounds.y, bounds.getMaxX(), bounds.getMaxY(), 1744822402, 1744822402);
                        class_2561 text = class_2561.method_43469("text.rei.release_export", export.getLocalizedName().method_27662().getString());
                        graphics.method_51448().pushMatrix();
                        graphics.method_51448().translate(bounds.getCenterX() - field_22793.method_27525(text) / 2f, bounds.getCenterY() - 4.5f);
                        graphics.method_51439(field_22793, text, 0, 0, 0xff000000, false);
                        graphics.method_51448().popMatrix();
                    } else {
                        graphics.method_25296(bounds.x, bounds.y, bounds.getMaxX(), bounds.getMaxY(), 1744830463, 1744830463);
                    }
                }
            }
        }
    }
    
    private Iterable<TabWidget> getTabs() {
        return Widgets.walk(widgets(), widget -> widget instanceof TabWidget);
    }
    
    @Override
    public boolean method_16803(class_11908 event) {
        ModifierKeyCode export = ConfigObject.getInstance().getExportImageKeybind();
        if (export.matchesKey(event.comp_4795(), event.comp_4796())) {
            if (checkExportDisplays()) return true;
        }
        return super.method_16803(event);
    }
    
    public int getCurrentTotalPages() {
        return getTotalPages(selectedCategoryIndex);
    }
    
    public int getTotalPages(int categoryIndex) {
        return class_3532.method_15384(categoryMap.get(categories.get(categoryIndex)).size() / (double) (getRecipesPerPage() + 1));
    }
    
    @Override
    public boolean method_25400(class_11905 event) {
        for (class_364 listener : method_25396())
            if (listener.method_25400(event))
                return true;
        return super.method_25400(event);
    }
    
    @Override
    public boolean method_25403(class_11909 event, double deltaX, double deltaY) {
        for (class_364 entry : method_25396())
            if (entry.method_25403(event, deltaX, deltaY))
                return true;
        return super.method_25403(event, deltaX, deltaY);
    }
    
    @Override
    public boolean method_25406(class_11909 event) {
        ModifierKeyCode export = ConfigObject.getInstance().getExportImageKeybind();
        if (export.matchesMouse(event.method_74245())) {
            if (checkExportDisplays()) return true;
        }
        for (class_364 entry : method_25396())
            if (entry.method_25406(event))
                return true;
        return super.method_25406(event);
    }
    
    private boolean checkExportDisplays() {
        for (Map.Entry<Rectangle, Pair<DisplaySpec, List<Widget>>> entry : recipeBounds.entrySet()) {
            Rectangle bounds = entry.getKey();
            if (bounds.contains(PointHelper.ofMouse())) {
                RecipeDisplayExporter.exportRecipeDisplay(bounds, entry.getValue().left(), entry.getValue().right(), true);
                return true;
            }
        }
        for (TabWidget tab : getTabs()) {
            Rectangle bounds = tab.getBounds();
            if (bounds.contains(PointHelper.ofMouse())) {
                field_22787.method_1507(new class_410(confirmed -> {
                    if (confirmed) {
                        for (DisplaySpec spec : categoryMap.getOrDefault(tab.category, Collections.emptyList())) {
                            Display display = spec.provideInternalDisplay();
                            int displayWidth = getCurrentCategory().getDisplayWidth(display);
                            int displayHeight = getCurrentCategory().getDisplayHeight();
                            final Rectangle displayBounds = new Rectangle(0, 0, displayWidth, displayHeight);
                            List<Widget> setupDisplay;
                            try {
                                setupDisplay = getCurrentCategoryView(display).setupDisplay(display, displayBounds);
                            } catch (Throwable throwable) {
                                throwable.printStackTrace();
                                setupDisplay = new ArrayList<>();
                                setupDisplay.add(Widgets.createRecipeBase(displayBounds).color(0xFFBB0000));
                                setupDisplay.add(Widgets.createLabel(new Point(displayBounds.getCenterX(), displayBounds.getCenterY() - 8), class_2561.method_43470("Failed to initiate setupDisplay")));
                                setupDisplay.add(Widgets.createLabel(new Point(displayBounds.getCenterX(), displayBounds.getCenterY() + 1), class_2561.method_43470("Check console for error")));
                            }
                            setupTags(setupDisplay);
                            transformFiltering(setupDisplay);
                            transformIngredientNotice(setupDisplay, ingredientStackToNotice);
                            transformResultNotice(setupDisplay, resultStackToNotice);
                            unifyIngredients(setupDisplay);
                            for (EntryWidget widget : Widgets.<EntryWidget>walk(widgets(), EntryWidget.class::isInstance)) {
                                widget.removeTagMatch = true;
                            }
                            
                            RecipeDisplayExporter.exportRecipeDisplay(displayBounds, spec, setupDisplay, false);
                        }
                        ExportRecipeIdentifierToast.addToast(class_1074.method_4662("msg.rei.exported_recipe"), class_1074.method_4662("msg.rei.exported_recipe.desc"));
                    }
                    field_22787.method_1507(null);
                }, class_2561.method_43469("text.rei.ask_to_export", tab.categoryName),
                        class_2561.method_43469("text.rei.ask_to_export.subtitle", categoryMap.getOrDefault(tab.category, Collections.emptyList()).size())));
            }
        }
        return false;
    }
    
    @Override
    public boolean method_25401(double mouseX, double mouseY, double amountX, double amountY) {
        REIRuntimeImpl.isWithinRecipeViewingScreen = true;
        for (class_364 listener : method_25396()) {
            if (listener.method_25401(mouseX, mouseY, amountX, amountY)) {
                REIRuntimeImpl.isWithinRecipeViewingScreen = false;
                return true;
            }
        }
        REIRuntimeImpl.isWithinRecipeViewingScreen = false;
        if (getBounds().contains(PointHelper.ofMouse())) {
            if (amountY > 0 && recipeBack.isEnabled())
                recipeBack.onClick();
            else if (amountY < 0 && recipeNext.isEnabled())
                recipeNext.onClick();
        }
        return super.method_25401(mouseX, mouseY, amountX, amountY);
    }
    
    @Override
    public boolean method_25402(class_11909 event, boolean doubleClick) {
        if (ConfigObject.getInstance().getNextPageKeybind().matchesMouse(event.method_74245())) {
            if (recipeNext.isEnabled())
                recipeNext.onClick();
            return recipeNext.isEnabled();
        } else if (ConfigObject.getInstance().getPreviousPageKeybind().matchesMouse(event.method_74245())) {
            if (recipeBack.isEnabled())
                recipeBack.onClick();
            return recipeBack.isEnabled();
        } else if (ConfigObject.getInstance().getPreviousScreenKeybind().matchesMouse(event.method_74245())) {
            if (REIRuntimeImpl.getInstance().hasLastDisplayScreen()) {
                field_22787.method_1507(REIRuntimeImpl.getInstance().getLastDisplayScreen());
            } else {
                field_22787.method_1507(REIRuntime.getInstance().getPreviousScreen());
            }
            return true;
        }
        return super.method_25402(event, doubleClick);
    }
    
    @Override
    public Optional<class_364> method_19355(double mouseX, double mouseY) {
        // Reverse iteration
        for (int i = widgets.size() - 1; i >= 0; i--) {
            Widget widget = widgets.get(i);
            if (widget.containsMouse(mouseX, mouseY)) {
                return Optional.of(widget);
            }
        }
        
        return Optional.empty();
    }
    
    public static class WorkstationSlotWidget extends EntryWidget {
        public WorkstationSlotWidget(int x, int y, EntryIngredient widgets) {
            super(new Point(x, y));
            entries(widgets);
            noBackground();
        }
        
        @Override
        public boolean containsMouse(double mouseX, double mouseY) {
            return getInnerBounds().contains(mouseX, mouseY);
        }
    }
}
