/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.api.common.util;

import com.google.common.collect.ImmutableList;
import dev.architectury.fluid.FluidStack;
import me.shedaniel.rei.api.common.display.basic.BasicDisplay;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.entry.type.EntryDefinition;
import me.shedaniel.rei.api.common.entry.type.EntryType;
import me.shedaniel.rei.api.common.entry.type.VanillaEntryTypes;
import me.shedaniel.rei.impl.Internals;
import me.shedaniel.rei.impl.common.InternalLogger;
import net.minecraft.class_10302;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_1935;
import net.minecraft.class_3611;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_6885;
import net.minecraft.class_7871;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;

public final class EntryIngredients {
    private EntryIngredients() {
    }
    
    public static EntryIngredient of(class_1935 stack) {
        return EntryIngredient.of(EntryStacks.of(stack));
    }
    
    public static EntryIngredient of(class_1935 stack, int amount) {
        return EntryIngredient.of(EntryStacks.of(stack, amount));
    }
    
    public static EntryIngredient of(class_1799 stack) {
        return EntryIngredient.of(EntryStacks.of(stack));
    }
    
    public static EntryIngredient of(class_3611 fluid) {
        return EntryIngredient.of(EntryStacks.of(fluid));
    }
    
    public static EntryIngredient of(class_3611 fluid, long amount) {
        return EntryIngredient.of(EntryStacks.of(fluid, amount));
    }
    
    public static EntryIngredient of(FluidStack stack) {
        return EntryIngredient.of(EntryStacks.of(stack));
    }
    
    public static EntryIngredient ofFluidHolder(class_6880<class_3611> fluid) {
        return EntryIngredient.of(EntryStacks.ofFluidHolder(fluid));
    }
    
    public static EntryIngredient ofFluidHolder(class_6880<class_3611> fluid, long amount) {
        return EntryIngredient.of(EntryStacks.ofFluidHolder(fluid, amount));
    }
    
    public static EntryIngredient ofItemHolder(class_6880<? extends class_1935> item) {
        return EntryIngredient.of(EntryStacks.ofItemHolder(item));
    }
    
    public static EntryIngredient ofItemHolder(class_6880<? extends class_1935> item, int amount) {
        return EntryIngredient.of(EntryStacks.ofItemHolder(item, amount));
    }
    
    public static <T> EntryIngredient of(EntryType<T> type, Collection<T> values) {
        return of(type.getDefinition(), values);
    }
    
    public static <T> EntryIngredient of(EntryDefinition<T> definition, Collection<T> values) {
        if (values.size() == 0) return EntryIngredient.empty();
        if (values.size() == 1) return EntryIngredient.of(EntryStack.of(definition, values.iterator().next()));
        EntryIngredient.Builder result = EntryIngredient.builder(values.size());
        for (T value : values) {
            result.add(EntryStack.of(definition, value));
        }
        return result.build();
    }
    
    public static <T> EntryIngredient from(Iterable<T> stacks, Function<T, ? extends EntryStack<?>> function) {
        if (stacks instanceof Collection<T> collection) {
            return from(collection, collection.size(), function);
        } else {
            if (!stacks.iterator().hasNext()) return EntryIngredient.empty();
            EntryIngredient.Builder result = EntryIngredient.builder();
            for (T t : stacks) {
                EntryStack<?> stack = function.apply(t);
                if (!stack.isEmpty()) {
                    result.add(stack);
                }
            }
            return result.build();
        }
    }
    
    public static <T> EntryIngredient from(Iterable<T> stacks, int size, Function<T, ? extends EntryStack<?>> function) {
        if (size == 0) return EntryIngredient.empty();
        if (size == 1) return EntryIngredient.of(function.apply(stacks.iterator().next()));
        EntryIngredient.Builder result = EntryIngredient.builder(size);
        for (T t : stacks) {
            EntryStack<?> stack = function.apply(t);
            if (!stack.isEmpty()) {
                result.add(stack);
            }
        }
        return result.build();
    }
    
    public static EntryIngredient ofItems(Collection<class_1935> stacks) {
        return ofItems(stacks, 1);
    }
    
    public static EntryIngredient ofItems(Collection<class_1935> stacks, int amount) {
        return from(stacks, stack -> EntryStacks.of(stack, amount));
    }
    
    public static EntryIngredient ofItemStacks(Collection<class_1799> stacks) {
        return of(VanillaEntryTypes.ITEM, stacks);
    }
    
    public static EntryIngredient ofIngredient(class_1856 ingredient) {
        return Internals.toEntryIngredient(ingredient);
    }
    
    public static List<EntryIngredient> ofIngredients(List<class_1856> ingredients) {
        if (ingredients.size() == 0) return Collections.emptyList();
        if (ingredients.size() == 1) {
            class_1856 ingredient = ingredients.get(0);
            EntryIngredient entryIngredient = ofIngredient(ingredient);
            if (entryIngredient.isEmpty()) return List.of();
            return List.of(entryIngredient);
        }
        boolean emptyFlag = true;
        List<EntryIngredient> result = new ArrayList<>(ingredients.size());
        for (int i = ingredients.size() - 1; i >= 0; i--) {
            class_1856 ingredient = ingredients.get(i);
            EntryIngredient entryIngredient = ofIngredient(ingredient);
            if (emptyFlag && entryIngredient.isEmpty()) continue;
            result.add(0, entryIngredient);
            emptyFlag = false;
        }
        return ImmutableList.copyOf(result);
    }
    
    public static <S, T> EntryIngredient ofTag(class_7871.class_7872 provider, class_6862<S> tagKey, Function<class_6880<S>, EntryStack<T>> mapper) {
        class_7871<S> getter = provider.method_46751(tagKey.comp_326());
        class_6885.class_6888<S> holders = getter.method_46733(tagKey).orElse(null);
        if (holders == null) return EntryIngredient.empty();
        return EntryIngredients.from(holders, holders.method_40247(), mapper);
    }
    
    public static <S, T> List<EntryIngredient> ofTags(class_7871.class_7872 provider, Iterable<class_6862<S>> tagKeys, Function<class_6880<S>, EntryStack<T>> mapper) {
        if (tagKeys instanceof Collection<?> collection && collection.isEmpty()) return Collections.emptyList();
        ImmutableList.Builder<EntryIngredient> ingredients = ImmutableList.builder();
        for (class_6862<S> tagKey : tagKeys) {
            ingredients.add(ofTag(provider, tagKey, mapper));
        }
        return ingredients.build();
    }
    
    public static <T extends class_1935> EntryIngredient ofItemTag(class_6862<T> tagKey) {
        return ofTag(BasicDisplay.registryAccess(), tagKey, EntryStacks::ofItemHolder);
    }
    
    public static EntryIngredient ofFluidTag(class_6862<class_3611> tagKey) {
        return ofTag(BasicDisplay.registryAccess(), tagKey, EntryStacks::ofFluidHolder);
    }
    
    public static <T extends class_1935> List<EntryIngredient> ofItemTags(Iterable<class_6862<T>> tagKey) {
        return ofTags(BasicDisplay.registryAccess(), tagKey, EntryStacks::ofItemHolder);
    }
    
    public static List<EntryIngredient> ofFluidTags(Iterable<class_6862<class_3611>> tagKey) {
        return ofTags(BasicDisplay.registryAccess(), tagKey, EntryStacks::ofFluidHolder);
    }
    
    public static EntryIngredient ofItemsHolderSet(class_6885<class_1792> stacks) {
        return stacks.method_40248().map(EntryIngredients::ofItemTag, holders -> from(holders, EntryStacks::ofItemHolder));
    }
    
    public static EntryIngredient ofFluidHolderSet(class_6885<class_3611> stacks) {
        return stacks.method_40248().map(EntryIngredients::ofFluidTag, holders -> from(holders, EntryStacks::ofFluidHolder));
    }
    
    public static EntryIngredient ofSlotDisplay(class_10302 slot) {
        return switch (slot) {
            case SlotDisplay.Empty $ -> EntryIngredient.empty();
            case SlotDisplay.ItemSlotDisplay s -> ofItemHolder(s.item());
            case SlotDisplay.ItemStackSlotDisplay s -> of(s.stack());
            case SlotDisplay.TagSlotDisplay s -> ofItemTag(s.tag());
            case SlotDisplay.Composite s -> {
                EntryIngredient.Builder builder = EntryIngredient.builder();
                for (SlotDisplay slotDisplay : s.contents()) {
                    builder.addAll(ofSlotDisplay(slotDisplay));
                }
                yield builder.build();
            }
            // TODO: Bad idea
            case SlotDisplay.AnyFuel s -> EntryIngredient.empty();
            default -> {
                RegistryAccess access = Internals.getRegistryAccess();
                try {
                    yield ofItemStacks(slot.resolveForStacks(new ContextMap.Builder()
                            .withParameter(SlotDisplayContext.REGISTRIES, access)
                            .create(SlotDisplayContext.CONTEXT)));
                } catch (Exception e) {
                    InternalLogger.getInstance().warn("Failed to resolve slot display: " + slot, e);
                    yield EntryIngredient.empty();
                }
            }
        };
    }
    
    public static List<EntryIngredient> ofSlotDisplays(Iterable<class_10302> slots) {
        if (slots instanceof Collection<?> collection && collection.isEmpty()) return Collections.emptyList();
        ImmutableList.Builder<EntryIngredient> ingredients = ImmutableList.builder();
        for (class_10302 slot : slots) {
            ingredients.add(ofSlotDisplay(slot));
        }
        return ingredients.build();
    }
    
    public static <T> boolean testFuzzy(EntryIngredient ingredient, EntryStack<T> stack) {
        for (EntryStack<?> ingredientStack : ingredient) {
            if (EntryStacks.equalsFuzzy(ingredientStack, stack)) {
                return true;
            }
        }
        
        return false;
    }
}
