/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.api.common.transfer;

import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import me.shedaniel.rei.api.common.util.CollectionUtils;
import net.minecraft.class_1661;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_6880;
import net.minecraft.class_9326;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

public class ItemRecipeFinder {
    private final Interner<ItemKey> keys = Interners.newWeakInterner();
    private final RecipeFinder<ItemKey, Ingredient> finder = new RecipeFinder<>();
    
    public boolean contains(class_1799 item) {
        return finder.contains(ofKey(item));
    }
    
    boolean containsAtLeast(class_1799 object, int i) {
        return finder.containsAtLeast(ofKey(object), i);
    }
    
    public void take(class_1799 item, int amount) {
        finder.take(ofKey(item), amount);
    }
    
    public void put(class_1799 item, int amount) {
        finder.put(ofKey(item), amount);
    }
    
    public void addNormalItem(class_1799 itemStack) {
        if (class_1661.method_61495(itemStack)) {
            this.addItem(itemStack);
        }
    }
    
    public void addItem(class_1799 itemStack) {
        this.addItem(itemStack, itemStack.method_7914());
    }
    
    public void addItem(class_1799 itemStack, int i) {
        if (!itemStack.method_7960()) {
            int j = Math.min(i, itemStack.method_7947());
            this.finder.put(ofKey(itemStack), j);
        }
    }
    
    public boolean findRecipe(List<List<class_1799>> list, int maxCrafts, @Nullable Consumer<class_1799> output) {
        return finder.findRecipe(toIngredients(list), maxCrafts, flatten(itemStack -> {
            if (output != null) {
                output.accept(itemStack);
            }
        }));
    }
    
    public int countRecipeCrafts(List<List<class_1799>> list, int maxCrafts, @Nullable Consumer<class_1799> output) {
        return finder.countRecipeCrafts(toIngredients(list), maxCrafts, flatten(itemStack -> {
            if (output != null) {
                output.accept(itemStack);
            }
        }));
    }
    
    private ItemKey ofKey(class_1799 itemStack) {
        return keys.intern(new ItemKey(itemStack.method_41409(), itemStack.method_57380()));
    }
    
    private Ingredient ofKeys(int index, List<class_1799> itemStack) {
        return new Ingredient(index, CollectionUtils.map(itemStack, this::ofKey));
    }
    
    private List<Ingredient> toIngredients(List<List<class_1799>> list) {
        List<Ingredient> ingredients = new ArrayList<>();
        
        for (int i = 0; i < list.size(); i++) {
            List<class_1799> stacks = list.get(i);
            if (!stacks.isEmpty()) {
                ingredients.add(ofKeys(i, stacks));
            }
        }
        
        return ingredients;
    }
    
    private static BiConsumer<ItemKey, Ingredient> flatten(Consumer<class_1799> consumer) {
        int[] lastIndex = {-1};
        return (itemKey, ingredient) -> {
            for (int i = lastIndex[0] + 1; i < ingredient.index(); i++) {
                consumer.accept(class_1799.field_8037);
            }
            consumer.accept(new class_1799(itemKey.item(), 1, itemKey.patch()));
            lastIndex[0] = ingredient.index();
        };
    }
    
    private record Ingredient(int index, List<ItemKey> elements) implements RecipeFinder.Ingredient<ItemKey> {
    }
    
    private record ItemKey(class_6880<class_1792> item, class_9326 patch) {
    }
}
