/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.widget.entrylist;

import com.google.common.collect.Lists;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.gui.config.SearchFieldLocation;
import me.shedaniel.rei.api.client.gui.widgets.Button;
import me.shedaniel.rei.api.client.gui.widgets.Widget;
import me.shedaniel.rei.api.client.gui.widgets.Widgets;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.util.CollectionUtils;
import me.shedaniel.rei.impl.client.ClientHelperImpl;
import me.shedaniel.rei.impl.client.config.ConfigManagerImpl;
import me.shedaniel.rei.impl.client.gui.InternalTextures;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.widget.BatchedEntryRendererManager;
import me.shedaniel.rei.impl.client.gui.widget.CachedEntryListRender;
import me.shedaniel.rei.impl.client.gui.widget.DefaultDisplayChoosePageWidget;
import me.shedaniel.rei.impl.client.gui.widget.EntryWidget;
import me.shedaniel.rei.impl.common.entry.type.collapsed.CollapsedStack;
import net.minecraft.class_124;
import net.minecraft.class_1921;
import net.minecraft.class_2561;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_437;
import java.util.*;
import java.util.stream.Stream;

public class PaginatedEntryListWidget extends CollapsingEntryListWidget {
    private Button leftButton, rightButton;
    private List<Widget> additionalWidgets;
    private List</*EntryStack<?> | EntryIngredient*/ Object> stacks = new ArrayList<>();
    private Object2IntMap<CollapsedStack> collapsedStackIndices = new Object2IntOpenHashMap<>();
    protected List<EntryListStackEntry> entries = Collections.emptyList();
    private int page;
    
    public int getPage() {
        return page;
    }
    
    public void setPage(int page) {
        this.page = page;
    }
    
    @Override
    protected void renderEntries(boolean fastEntryRendering, class_332 graphics, int mouseX, int mouseY, float delta) {
        this.leftButton.setEnabled(getTotalPages() > 1);
        this.rightButton.setEnabled(getTotalPages() > 1);
        
        if (ConfigObject.getInstance().doesCacheEntryRendering()) {
            for (EntryListStackEntry entry : entries) {
                CollapsedStack collapsedStack = entry.getCollapsedStack();
                if (collapsedStack != null && !collapsedStack.isExpanded()) {
                    continue;
                }
                
                if (entry.our == null) {
                    CachedEntryListRender.Sprite sprite = CachedEntryListRender.get(entry.getCurrentEntry());
                    if (sprite != null) {
                        CachingEntryRenderer renderer = new CachingEntryRenderer(sprite);
                        entry.our = entry.getCurrentEntry().copy().cast().withRenderer(stack -> renderer);
                    }
                }
            }
        }
        
        BatchedEntryRendererManager<EntryListStackEntry> manager = new BatchedEntryRendererManager<>();
        if (manager.isFastEntryRendering()) {
            for (EntryListStackEntry entry : entries) {
                CollapsedStack collapsedStack = entry.getCollapsedStack();
                if (collapsedStack != null && !collapsedStack.isExpanded()) {
                    manager.addSlow(entry);
                } else {
                    manager.add(entry);
                }
            }
        } else {
            manager.addAllSlow(entries);
        }
        manager.render(debugger.debugTime, debugger.size, debugger.time, graphics, mouseX, mouseY, delta);
        
        new CollapsedEntriesBorderRenderer().render(graphics, entries, collapsedStackIndices);
        
        for (Widget widget : additionalWidgets) {
            widget.method_25394(graphics, mouseX, mouseY, delta);
        }
    }
    
    public int getTotalPages() {
        return getTotalPages(entries);
    }
    
    public int getTotalPages(List<?> entries) {
        return class_3532.method_15386(stacks.size() / (float) entries.size());
    }
    
    @Override
    protected void updateEntries(int entrySize, boolean zoomed) {
        page = Math.max(page, 0);
        List<EntryListStackEntry> entries = Lists.newArrayList();
        int width = innerBounds.width / entrySize;
        int height = innerBounds.height / entrySize;
        for (int currentY = 0; currentY < height; currentY++) {
            for (int currentX = 0; currentX < width; currentX++) {
                int slotX = currentX * entrySize + innerBounds.x;
                int slotY = currentY * entrySize + innerBounds.y;
                if (notSteppingOnExclusionZones(slotX - 1, slotY - 1, entrySize, entrySize)) {
                    entries.add((EntryListStackEntry) new EntryListStackEntry(this, slotX, slotY, entrySize, zoomed).noBackground());
                }
            }
        }
        page = class_3532.method_15340(page, 0, getTotalPages(entries) - 1);
        int skip = Math.max(0, page * entries.size());
        List</*EntryStack<?> | List<EntryStack<?>>*/ Object> subList = stacks.subList(skip, Math.min(stacks.size(), skip + entries.size()));
        Int2ObjectMap<CollapsedStack> indexedCollapsedStack = getCollapsedStackIndexed();
        Set<CollapsedStack> collapsedStacks = new LinkedHashSet<>();
        for (int i = 0; i < subList.size(); i++) {
            /*EntryStack<?> | List<EntryStack<?>>*/ Object stack = subList.get(i);
            EntryListStackEntry entry = entries.get(i + Math.max(0, -page * entries.size()));
            entry.clearStacks();
            
            if (stack instanceof EntryStack<?> entryStack) {
                entry.entry(entryStack);
            } else {
                entry.entries((List<EntryStack<?>>) stack);
            }
            
            CollapsedStack collapsedStack = indexedCollapsedStack.get(i + skip);
            if (collapsedStack != null && collapsedStack.getIngredient().size() > 1) {
                entry.collapsed(collapsedStack);
                collapsedStacks.add(collapsedStack);
            } else {
                entry.collapsed(null);
            }
        }
        this.entries = entries;
        this.collapsedStackIndices = new Object2IntOpenHashMap<>();
        int index = 0;
        for (CollapsedStack stack : collapsedStacks) {
            collapsedStackIndices.put(stack, index++);
        }
    }
    
    @Override
    public List</*EntryStack<?> | List<EntryStack<?>>*/ Object> getStacks() {
        return stacks;
    }
    
    @Override
    public void setStacks(List</*EntryStack<?> | List<EntryStack<?>>*/ Object> stacks) {
        this.stacks = stacks;
    }
    
    @Override
    public Stream<EntryStack<?>> getEntries() {
        return entries.stream().map(EntryWidget::getCurrentEntry);
    }
    
    @Override
    protected List<EntryListStackEntry> getEntryWidgets() {
        return entries;
    }
    
    @Override
    public List<? extends Widget> method_25396() {
        return CollectionUtils.concatUnmodifiable(super.method_25396(), additionalWidgets);
    }
    
    @Override
    public void init(ScreenOverlayImpl overlay) {
        Rectangle overlayBounds = overlay.getBounds();
        this.additionalWidgets = new ArrayList<>();
        this.leftButton = Widgets.createButton(new Rectangle(overlayBounds.x, overlayBounds.y + (ConfigObject.getInstance().getSearchFieldLocation() == SearchFieldLocation.TOP_SIDE ? 24 : 0) + 5, 16, 16), class_2561.method_43470(""))
                .onClick(button -> {
                    setPage(getPage() - 1);
                    if (getPage() < 0)
                        setPage(getTotalPages() - 1);
                    updateEntriesPosition();
                })
                .containsMousePredicate((button, point) -> button.getBounds().contains(point) && overlay.isNotInExclusionZones(point.x, point.y))
                .tooltipLine(class_2561.method_43471("text.rei.previous_page"))
                .focusable(false);
        this.additionalWidgets.add(leftButton);
        this.additionalWidgets.add(Widgets.createDrawableWidget((graphics, mouseX, mouseY, delta) -> {
            Rectangle bounds = leftButton.getBounds();
            graphics.method_51448().method_22903();
            graphics.method_51448().method_46416(0, 0, 1);
            graphics.method_25290(class_1921::method_62277, InternalTextures.ARROW_LEFT_TEXTURE, bounds.x + 4, bounds.y + 4, 0, 0, 8, 8, 8, 8);
            graphics.method_51448().method_22909();
        }));
        this.rightButton = Widgets.createButton(new Rectangle(overlayBounds.getMaxX() - 18, overlayBounds.y + (ConfigObject.getInstance().getSearchFieldLocation() == SearchFieldLocation.TOP_SIDE ? 24 : 0) + 5, 16, 16), class_2561.method_43471(""))
                .onClick(button -> {
                    setPage(getPage() + 1);
                    if (getPage() >= getTotalPages())
                        setPage(0);
                    updateEntriesPosition();
                })
                .containsMousePredicate((button, point) -> button.getBounds().contains(point) && overlay.isNotInExclusionZones(point.x, point.y))
                .tooltipLine(class_2561.method_43471("text.rei.next_page"))
                .focusable(false);
        this.additionalWidgets.add(rightButton);
        this.additionalWidgets.add(Widgets.createDrawableWidget((graphics, mouseX, mouseY, delta) -> {
            Rectangle bounds = rightButton.getBounds();
            graphics.method_51448().method_22903();
            graphics.method_51448().method_46416(0, 0, 1);
            graphics.method_25290(class_1921::method_62277, InternalTextures.ARROW_RIGHT_TEXTURE, bounds.x + 4, bounds.y + 4, 0, 0, 8, 8, 8, 8);
            graphics.method_51448().method_22909();
        }));
        this.additionalWidgets.add(Widgets.createClickableLabel(new Point(overlayBounds.x + (overlayBounds.width / 2), overlayBounds.y + (ConfigObject.getInstance().getSearchFieldLocation() == SearchFieldLocation.TOP_SIDE ? 24 : 0) + 10), class_2561.method_43473(), label -> {
            if (!class_437.method_25442()) {
                setPage(0);
                updateEntriesPosition();
            } else {
                ScreenOverlayImpl.getInstance().choosePageWidget = new DefaultDisplayChoosePageWidget(page -> {
                    setPage(page);
                    updateEntriesPosition();
                }, getPage(), getTotalPages());
            }
        }).tooltip(class_2561.method_43471("text.rei.go_back_first_page"), class_2561.method_43470(" "), class_2561.method_43469("text.rei.shift_click_to", class_2561.method_43471("text.rei.choose_page")).method_27692(class_124.field_1080)).focusable(false).onRender((matrices, label) -> {
            label.setClickable(getTotalPages() > 1);
            label.setMessage(class_2561.method_43470(String.format("%s/%s", getPage() + 1, Math.max(getTotalPages(), 1))));
        }).rainbow(new Random().nextFloat() < 1.0E-4D || ClientHelperImpl.getInstance().isAprilFools.method_15332() || ConfigManagerImpl.getInstance().getConfig().appearance.rainbow));
    }
    
    @Override
    public boolean method_25401(double mouseX, double mouseY, double amountX, double amountY) {
        if (super.method_25401(mouseX, mouseY, amountX, amountY)) return true;
        if (!class_437.method_25441()) {
            if (amountY > 0 && leftButton.isEnabled())
                leftButton.onClick();
            else if (amountY < 0 && rightButton.isEnabled())
                rightButton.onClick();
            else
                return false;
            return true;
        }
        return false;
    }
}
