/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.widget.entrylist;

import me.shedaniel.clothconfig2.api.animator.NumberAnimator;
import me.shedaniel.clothconfig2.api.animator.ValueAnimator;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.ClientHelper;
import me.shedaniel.rei.api.client.REIRuntime;
import me.shedaniel.rei.api.client.config.ConfigManager;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.gui.drag.DraggableStack;
import me.shedaniel.rei.api.client.gui.drag.DraggableStackVisitorWidget;
import me.shedaniel.rei.api.client.gui.drag.DraggedAcceptorResult;
import me.shedaniel.rei.api.client.gui.drag.DraggingContext;
import me.shedaniel.rei.api.client.gui.screen.DisplayScreen;
import me.shedaniel.rei.api.client.gui.widgets.Slot;
import me.shedaniel.rei.api.client.gui.widgets.Tooltip;
import me.shedaniel.rei.api.client.gui.widgets.Widget;
import me.shedaniel.rei.api.client.gui.widgets.WidgetWithBounds;
import me.shedaniel.rei.api.client.overlay.OverlayListWidget;
import me.shedaniel.rei.api.client.overlay.ScreenOverlay;
import me.shedaniel.rei.api.client.registry.screen.OverlayDecider;
import me.shedaniel.rei.api.client.registry.screen.ScreenRegistry;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.entry.type.VanillaEntryTypes;
import me.shedaniel.rei.api.common.util.EntryStacks;
import me.shedaniel.rei.impl.client.ClientHelperImpl;
import me.shedaniel.rei.impl.client.config.ConfigManagerImpl;
import me.shedaniel.rei.impl.client.config.ConfigObjectImpl;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.widget.EntryWidget;
import me.shedaniel.rei.impl.client.gui.widget.favorites.FavoritesListWidget;
import me.shedaniel.rei.impl.client.gui.widget.region.RegionRenderingDebugger;
import net.minecraft.class_1269;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_437;
import net.minecraft.class_746;
import org.jetbrains.annotations.ApiStatus;

import java.util.List;

@SuppressWarnings("UnstableApiUsage")
@ApiStatus.Internal
public abstract class EntryListWidget extends WidgetWithBounds implements OverlayListWidget, DraggableStackVisitorWidget {
    private static final int SIZE = 18;
    protected final RegionRenderingDebugger debugger = new RegionRenderingDebugger();
    protected Rectangle bounds, innerBounds;
    public final NumberAnimator<Double> scaleIndicator = ValueAnimator.ofDouble(0.0D)
            .withConvention(() -> 0.0D, 8000);
    
    public static int entrySize() {
        return class_3532.method_15384(SIZE * ConfigObject.getInstance().getEntrySize());
    }
    
    public static boolean notSteppingOnExclusionZones(int left, int top, int width, int height) {
        class_310 instance = class_310.method_1551();
        for (OverlayDecider decider : ScreenRegistry.getInstance().getDeciders(instance.field_1755)) {
            class_1269 fit = canItemSlotWidgetFit(left, top, width, height, decider);
            if (fit != class_1269.field_5811)
                return fit == class_1269.field_5812;
        }
        return true;
    }
    
    private static class_1269 canItemSlotWidgetFit(int left, int top, int width, int height, OverlayDecider decider) {
        class_1269 fit;
        fit = decider.isInZone(left, top);
        if (fit != class_1269.field_5811)
            return fit;
        fit = decider.isInZone(left + width, top);
        if (fit != class_1269.field_5811)
            return fit;
        fit = decider.isInZone(left, top + height);
        if (fit != class_1269.field_5811)
            return fit;
        fit = decider.isInZone(left + width, top + height);
        return fit;
    }
    
    private boolean containsChecked(Point point, boolean inner) {
        return containsChecked(point.x, point.y, inner);
    }
    
    @Override
    public boolean containsMouse(double mouseX, double mouseY) {
        return hasSpace() && super.containsMouse(mouseX, mouseY);
    }
    
    public boolean innerContainsMouse(double mouseX, double mouseY) {
        return hasSpace() && innerBounds.contains(mouseX, mouseY);
    }
    
    public boolean containsChecked(double x, double y, boolean inner) {
        if (inner) {
            if (!innerContainsMouse(x, y)) return false;
        } else {
            if (!containsMouse(x, y)) return false;
        }
        class_310 instance = class_310.method_1551();
        for (OverlayDecider decider : ScreenRegistry.getInstance().getDeciders(instance.field_1755)) {
            class_1269 result = decider.isInZone(x, y);
            if (result != class_1269.field_5811)
                return result == class_1269.field_5812;
        }
        return true;
    }
    
    private static Rectangle updateInnerBounds(Rectangle bounds) {
        bounds = bounds.clone();
        int heightReduction = (int) Math.round(bounds.height * (1 - ConfigObject.getInstance().getVerticalEntriesBoundariesPercentage()));
        bounds.y += (int) Math.round(heightReduction * ConfigObject.getInstance().getVerticalEntriesBoundariesAlignments());
        bounds.height -= heightReduction;
        int maxHeight = (int) Math.ceil(entrySize() * ConfigObject.getInstance().getVerticalEntriesBoundariesRows() + entrySize() * 0.75);
        if (bounds.height > maxHeight) {
            bounds.y += (int) Math.round((bounds.height - maxHeight) * ConfigObject.getInstance().getVerticalEntriesBoundariesAlignments());
            bounds.height = maxHeight;
        }
        int entrySize = entrySize();
        if (ConfigObject.getInstance().isEntryListWidgetScrolled()) {
            int width = Math.max(class_3532.method_15375((bounds.width - 2 - 6) / (float) entrySize), 1);
            if (ConfigObject.getInstance().isLeftHandSidePanel()) {
                return new Rectangle((int) (bounds.getCenterX() - width * (entrySize / 2f) + 3), bounds.y, width * entrySize, bounds.height);
            }
            return new Rectangle((int) (bounds.getCenterX() - width * (entrySize / 2f) - 3), bounds.y, width * entrySize, bounds.height);
        } else {
            int width = Math.max(class_3532.method_15375((bounds.width - 2) / (float) entrySize), 1);
            int height = Math.max(class_3532.method_15375((bounds.height - 2) / (float) entrySize), 1);
            return new Rectangle((int) (bounds.getCenterX() - width * (entrySize / 2f)), (int) (bounds.getCenterY() - height * (entrySize / 2f)), width * entrySize, height * entrySize);
        }
    }
    
    @Override
    public DraggedAcceptorResult acceptDraggedStack(DraggingContext<class_437> context, DraggableStack stack) {
        if (innerBounds.contains(context.getCurrentPosition())) {
            context.renderToVoid(stack);
            return DraggedAcceptorResult.CONSUMED;
        } else {
            return DraggedAcceptorResult.PASS;
        }
    }
    
    @Override
    public boolean method_25401(double mouseX, double mouseY, double amountX, double amountY) {
        if (containsChecked(mouseX, mouseY, false) && amountY != 0) {
            if (class_437.method_25441()) {
                ConfigObjectImpl config = ConfigManagerImpl.getInstance().getConfig();
                scaleIndicator.setAs(10.0D);
                if (config.setEntrySize(config.getEntrySize() + Double.compare(amountY, 0) * 0.05)) {
                    ConfigManager.getInstance().saveConfig();
                    REIRuntime.getInstance().getOverlay().ifPresent(ScreenOverlay::queueReloadOverlay);
                    return true;
                }
            }
        }
        
        return super.method_25401(mouseX, mouseY, amountX, amountY);
    }
    
    @Override
    public Rectangle getBounds() {
        return bounds;
    }
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        if (!hasSpace()) return;
        
        boolean fastEntryRendering = ConfigObject.getInstance().doesFastEntryRendering();
        renderEntries(fastEntryRendering, graphics, mouseX, mouseY, delta);
        
        debugger.render(graphics, bounds.x, bounds.y, delta);
        
        if (containsChecked(mouseX, mouseY, false) && ClientHelper.getInstance().isCheating() && !(class_310.method_1551().field_1755 instanceof DisplayScreen) && !minecraft.field_1724.field_7512.method_34255().method_7960() && ClientHelperImpl.getInstance().canDeleteItems()) {
            EntryStack<?> stack = EntryStacks.of(minecraft.field_1724.field_7512.method_34255().method_7972());
            if (stack.getType() != VanillaEntryTypes.ITEM) {
                EntryStack<class_1799> cheatsAs = stack.cheatsAs();
                stack = cheatsAs.isEmpty() ? stack : cheatsAs;
            }
            for (Widget child : method_25396()) {
                if (child.containsMouse(mouseX, mouseY) && child instanceof EntryWidget widget) {
                    if (widget.cancelDeleteItems(stack)) {
                        return;
                    }
                }
            }
            Tooltip.create(class_2561.method_43471("text.rei.delete_items")).queue();
        }
        
        scaleIndicator.update(delta);
        if (scaleIndicator.value() > 0.04) {
            graphics.method_51448().method_22903();
            graphics.method_51448().method_46416(0, 0, 500);
            class_2561 component = class_2561.method_43470(Math.round(ConfigObject.getInstance().getEntrySize() * 100) + "%");
            int width = font.method_27525(component);
            int backgroundColor = ((int) Math.round(0xa0 * class_3532.method_15350(scaleIndicator.value(), 0.0, 1.0))) << 24;
            int textColor = ((int) Math.round(0xdd * class_3532.method_15350(scaleIndicator.value(), 0.0, 1.0))) << 24;
            graphics.method_25296(bounds.getCenterX() - width / 2 - 2, bounds.getCenterY() - 6, bounds.getCenterX() + width / 2 + 2, bounds.getCenterY() + 6, backgroundColor, backgroundColor);
            graphics.method_51439(font, component, bounds.getCenterX() - width / 2, bounds.getCenterY() - 4, 0xFFFFFF | textColor, false);
            graphics.method_51448().method_22909();
        }
    }
    
    protected abstract void renderEntries(boolean fastEntryRendering, class_332 graphics, int mouseX, int mouseY, float delta);
    
    @Override
    public boolean method_25404(int keyCode, int scanCode, int modifiers) {
        if (containsChecked(mouse(), false))
            for (Widget widget : getEntryWidgets())
                if (widget.method_25404(keyCode, scanCode, modifiers))
                    return true;
        return false;
    }
    
    public void updateArea(Rectangle bounds, String searchTerm) {
        this.bounds = REIRuntime.getInstance().calculateEntryListArea(bounds);
        FavoritesListWidget favoritesListWidget = ScreenOverlayImpl.getFavoritesListWidget();
        if (favoritesListWidget != null) {
            favoritesListWidget.updateFavoritesBounds(searchTerm);
        }
        if (ConfigObject.getInstance().isFavoritesEnabled() && favoritesListWidget == null) {
            updateSearch(searchTerm, true);
        } else {
            updateEntriesPosition();
        }
    }
    
    public boolean hasSpace() {
        int entrySize = entrySize();
        int width = innerBounds.width / entrySize;
        int height = innerBounds.height / entrySize;
        return width * height > 0;
    }
    
    public void updateEntriesPosition() {
        int entrySize = entrySize();
        boolean zoomed = ConfigObject.getInstance().isFocusModeZoomed();
        this.innerBounds = updateInnerBounds(bounds);
        updateEntries(entrySize, zoomed);
        FavoritesListWidget favoritesListWidget = ScreenOverlayImpl.getFavoritesListWidget();
        if (favoritesListWidget != null) {
            favoritesListWidget.getSystemRegion().updateEntriesPosition(entry -> true);
            favoritesListWidget.getRegion().updateEntriesPosition(entry -> true);
        }
    }
    
    protected abstract void updateEntries(int entrySize, boolean zoomed);
    
    public abstract boolean isEmpty();
    
    protected abstract void setCollapsedStacks(List</*EntryStack<?> | CollapsedStack*/ Object> stacks);
    
    public void updateSearch(String searchTerm, boolean ignoreLastSearch) {
        EntryListSearchManager.INSTANCE.update(searchTerm, ignoreLastSearch, stacks -> {
            setCollapsedStacks(stacks);
            updateEntriesPosition();
        });
        debugger.debugTime = ConfigObject.getInstance().doDebugRenderTimeRequired();
        FavoritesListWidget favorites = ScreenOverlayImpl.getFavoritesListWidget();
        if (favorites != null) {
            favorites.updateSearch();
        }
    }
    
    @Override
    public List<? extends Widget> method_25396() {
        return getEntryWidgets();
    }
    
    @Override
    public boolean method_25402(double mouseX, double mouseY, int button) {
        if (!hasSpace()) return false;
        for (Widget widget : method_25396())
            if (widget.method_25402(mouseX, mouseY, button))
                return true;
        return false;
    }
    
    @Override
    public boolean method_25406(double mouseX, double mouseY, int button) {
        if (containsChecked(mouseX, mouseY, false)) {
            class_746 player = minecraft.field_1724;
            if (ClientHelper.getInstance().isCheating() && !(class_310.method_1551().field_1755 instanceof DisplayScreen) && player != null && player.field_7512 != null && !player.field_7512.method_34255().method_7960() && ClientHelperImpl.getInstance().canDeleteItems()) {
                EntryStack<?> stack = EntryStacks.of(minecraft.field_1724.field_7512.method_34255().method_7972());
                if (stack.getType() != VanillaEntryTypes.ITEM) {
                    EntryStack<class_1799> cheatsAs = stack.cheatsAs();
                    stack = cheatsAs.isEmpty() ? stack : cheatsAs;
                }
                boolean canDelete = true;
                
                for (Widget child : method_25396()) {
                    if (child.containsMouse(mouseX, mouseY) && child instanceof EntryWidget widget) {
                        if (widget.cancelDeleteItems(stack)) {
                            canDelete = false;
                            break;
                        }
                    }
                }
                
                if (canDelete) {
                    ClientHelper.getInstance().sendDeletePacket();
                    return true;
                }
            }
            for (Widget widget : method_25396())
                if (widget.method_25406(mouseX, mouseY, button))
                    return true;
        }
        return false;
    }
    
    @Override
    public EntryStack<?> getFocusedStack() {
        Point mouse = mouse();
        if (containsChecked(mouse, false)) {
            for (Slot entry : getEntryWidgets()) {
                EntryStack<?> currentEntry = entry.getCurrentEntry();
                if (!currentEntry.isEmpty() && entry.containsMouse(mouse)) {
                    return currentEntry.copy();
                }
            }
        }
        return EntryStack.empty();
    }
    
    protected abstract List<EntryListStackEntry> getEntryWidgets();
    
    public void init(ScreenOverlayImpl overlay) {
    }
}
