/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.modules;

import com.google.common.collect.Lists;
import me.shedaniel.clothconfig2.ClothConfigInitializer;
import me.shedaniel.clothconfig2.api.animator.ValueAnimator;
import me.shedaniel.clothconfig2.api.scroll.ScrollingContainer;
import me.shedaniel.math.FloatingRectangle;
import me.shedaniel.math.Point;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.REIRuntime;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.favorites.FavoriteMenuEntry;
import me.shedaniel.rei.api.client.gui.widgets.WidgetWithBounds;
import me.shedaniel.rei.impl.client.gui.ScreenOverlayImpl;
import me.shedaniel.rei.impl.client.gui.modules.entries.SubMenuEntry;
import me.shedaniel.rei.impl.client.gui.widget.LateRenderable;
import net.minecraft.class_310;
import net.minecraft.class_332;
import org.jetbrains.annotations.ApiStatus;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;

@ApiStatus.Internal
public class Menu extends WidgetWithBounds implements LateRenderable {
    public final Point menuStartPoint;
    public final boolean facingRight;
    public final boolean facingDownwards;
    public final ValueAnimator<FloatingRectangle> bounds = ValueAnimator.ofFloatingRectangle();
    private final List<FavoriteMenuEntry> entries = Lists.newArrayList();
    public final ScrollingContainer scrolling = new ScrollingContainer() {
        @Override
        public int getMaxScrollHeight() {
            int i = 0;
            for (FavoriteMenuEntry entry : method_25396()) {
                i += entry.getEntryHeight();
            }
            return i;
        }
        
        @Override
        public Rectangle getBounds() {
            return Menu.this.getInnerBounds();
        }
        
        @Override
        public boolean hasScrollBar() {
            return Menu.this.hasScrollBar();
        }
    };
    
    public Menu(Rectangle menuStart, Collection<FavoriteMenuEntry> entries, boolean sort) {
        buildEntries(entries, sort);
        int fullWidth = class_310.method_1551().field_1755.field_22789;
        int fullHeight = class_310.method_1551().field_1755.field_22790;
        boolean facingRight = true;
        this.facingDownwards = fullHeight - menuStart.getMaxY() > menuStart.y;
        int y = facingDownwards ? menuStart.getMaxY() : menuStart.y - (scrolling.getMaxScrollHeight() + 2);
        boolean hasScrollBar = scrolling.getMaxScrollHeight() > getInnerHeight(y);
        int menuWidth = getMaxEntryWidth() + 2 + (hasScrollBar ? 6 : 0);
        if (facingRight && fullWidth - menuStart.getMaxX() < menuWidth + 10) {
            facingRight = false;
        } else if (!facingRight && menuStart.x < menuWidth + 10) {
            facingRight = true;
        }
        this.facingRight = facingRight;
        int x = facingRight ? menuStart.x : menuStart.getMaxX() - (getMaxEntryWidth() + 2 + (hasScrollBar ? 6 : 0));
        this.menuStartPoint = new Point(x, y);
        Rectangle createBounds = createBounds();
        this.bounds.setAs(new FloatingRectangle(facingRight ? createBounds.x : createBounds.getMaxX(), facingDownwards ? createBounds.y : createBounds.getMaxY(), 0.1, 0.1));
    }
    
    private void buildEntries(Collection<FavoriteMenuEntry> entries, boolean sort) {
        this.entries.clear();
        this.entries.addAll(entries);
        if (sort) {
            this.entries.sort(Comparator.comparing(entry -> entry instanceof SubMenuEntry ? 0 : 1)
                    .thenComparing(entry -> entry instanceof SubMenuEntry menuEntry ? menuEntry.text.getString() : ""));
        }
        for (FavoriteMenuEntry entry : this.entries) {
            entry.closeMenu = ScreenOverlayImpl.getInstance().menuAccess()::close;
            if (entry instanceof SubMenuEntry menuEntry) {
                menuEntry.setParent(this);
            }
        }
    }
    
    @Override
    public Rectangle getBounds() {
        return bounds.value().getBounds();
    }
    
    public Rectangle createBounds() {
        return new Rectangle(menuStartPoint.x, menuStartPoint.y, getMaxEntryWidth() + 2 + (hasScrollBar() ? 6 : 0), getInnerHeight(menuStartPoint.y) + 2);
    }
    
    public Rectangle getInnerBounds() {
        Rectangle rectangle = bounds.value().getBounds();
        return new Rectangle(rectangle.x + 1, rectangle.y + 1, rectangle.width - 2, rectangle.height - 2);
    }
    
    public boolean hasScrollBar() {
        return scrolling.getMaxScrollHeight() > getInnerHeight(menuStartPoint.y);
    }
    
    public int getInnerHeight(int y) {
        return Math.min(scrolling.getMaxScrollHeight(), minecraft.field_1755.field_22790 - 20 - y);
    }
    
    public int getMaxEntryWidth() {
        int i = 0;
        for (FavoriteMenuEntry entry : method_25396()) {
            if (entry.getEntryWidth() > i)
                i = entry.getEntryWidth();
        }
        return Math.max(10, i);
    }
    
    @Override
    public void method_25394(class_332 graphics, int mouseX, int mouseY, float delta) {
        this.bounds.setTo(createBounds().getFloatingBounds(), ConfigObject.getInstance().isReducedMotion() ? 0 : 300);
        this.bounds.update(delta);
        Rectangle bounds = getBounds();
        Rectangle innerBounds = getInnerBounds();
        graphics.method_25294(bounds.x, bounds.y, bounds.getMaxX(), bounds.getMaxY(), containsMouse(mouseX, mouseY) ? (REIRuntime.getInstance().isDarkThemeEnabled() ? -17587 : -1) : -6250336);
        graphics.method_25294(innerBounds.x, innerBounds.y, innerBounds.getMaxX(), innerBounds.getMaxY(), -16777216);
        boolean contains = innerBounds.contains(mouseX, mouseY);
        FavoriteMenuEntry focused = method_25399() instanceof FavoriteMenuEntry menuEntry ? menuEntry : null;
        int currentY = innerBounds.y - scrolling.scrollAmountInt();
        for (FavoriteMenuEntry child : method_25396()) {
            boolean containsMouse = contains && mouseY >= currentY && mouseY < currentY + child.getEntryHeight();
            if (containsMouse) {
                focused = child;
            }
            currentY += child.getEntryHeight();
        }
        currentY = innerBounds.y - scrolling.scrollAmountInt();
        graphics.method_44379(scrolling.getScissorBounds().x, scrolling.getScissorBounds().y, scrolling.getScissorBounds().getMaxX(), scrolling.getScissorBounds().getMaxY());
        for (FavoriteMenuEntry child : method_25396()) {
            boolean rendering = currentY + child.getEntryHeight() >= innerBounds.y && currentY <= innerBounds.getMaxY();
            boolean containsMouse = contains && mouseY >= currentY && mouseY < currentY + child.getEntryHeight();
            child.updateInformation(innerBounds.x, currentY, focused == child || containsMouse, containsMouse, rendering, getMaxEntryWidth());
            if (rendering) {
                child.method_25394(graphics, mouseX, mouseY, delta);
            }
            currentY += child.getEntryHeight();
        }
        graphics.method_44380();
        method_25395(focused);
        scrolling.renderScrollBar(graphics, 0, REIRuntime.getInstance().isDarkThemeEnabled() ? 0.8f : 1f);
        scrolling.updatePosition(delta);
    }
    
    @Override
    public boolean method_25402(double mouseX, double mouseY, int button) {
        if (scrolling.updateDraggingState(mouseX, mouseY, button))
            return true;
        return super.method_25402(mouseX, mouseY, button) || getInnerBounds().contains(mouseX, mouseY);
    }
    
    @Override
    public boolean method_25403(double mouseX, double mouseY, int button, double deltaX, double deltaY) {
        if (scrolling.mouseDragged(mouseX, mouseY, button, deltaX, deltaY))
            return true;
        return super.method_25403(mouseX, mouseY, button, deltaX, deltaY);
    }
    
    @Override
    public boolean method_25401(double mouseX, double mouseY, double amountX, double amountY) {
        if (getInnerBounds().contains(mouseX, mouseY)) {
            scrolling.offset(ClothConfigInitializer.getScrollStep() * -amountY, true);
            return true;
        }
        for (FavoriteMenuEntry child : method_25396()) {
            if (child instanceof SubMenuEntry) {
                if (child.method_25401(mouseX, mouseY, amountX, amountY))
                    return true;
            }
        }
        return super.method_25401(mouseX, mouseY, amountX, amountY);
    }
    
    @Override
    public boolean containsMouse(double mouseX, double mouseY) {
        if (super.containsMouse(mouseX, mouseY)) return true;
        for (FavoriteMenuEntry child : method_25396()) {
            if (child.containsMouse(mouseX, mouseY)) {
                return true;
            }
        }
        return false;
    }
    
    @Override
    public List<FavoriteMenuEntry> method_25396() {
        return entries;
    }
}
