/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.api.client.entry.renderer;

import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.common.entry.EntryStack;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_332;
import net.minecraft.class_4587;
import net.minecraft.class_4597;

/**
 * A batched renderer for rendering a lot of {@link EntryStack} at once with better performance.
 *
 * @param <T> the entry type
 * @param <E> the type of extra data returned in {@link #getExtraData(EntryStack)}
 */
@Environment(EnvType.CLIENT)
public interface BatchedEntryRenderer<T, E> extends EntryRenderer<T> {
    default boolean isBatched(EntryStack<T> entry) {
        return true;
    }
    
    /**
     * Returns extra data to be passed to various rendering methods.
     *
     * @param entry the stack
     * @return the extra data
     */
    E getExtraData(EntryStack<T> entry);
    
    /**
     * Returns a batch identifier, stacks with the same batch identifier will be grouped together
     * into a batch.
     *
     * @param entry     the stack
     * @param bounds    the bounds of the entry
     * @param extraData the extra data returned from {@link #getExtraData(EntryStack)}
     * @return the batch identifier
     */
    int getBatchIdentifier(EntryStack<T> entry, Rectangle bounds, E extraData);
    
    /**
     * Modifies the {@link class_4587} passed tp various batch rendering methods.
     *
     * @param matrices the matrix stack
     * @return the modified matrix stack, could be an entirely different stack
     */
    default class_4587 batchModifyMatrices(class_4587 matrices) {
        return matrices;
    }
    
    /**
     * Starts the batch rendering, used to setup states, only called once with every batch.
     *
     * @param entry     the first entry in the batch
     * @param extraData the extra data returned from {@link #getExtraData(EntryStack)}
     * @param graphics  the graphics context
     * @param delta     the tick delta
     */
    void startBatch(EntryStack<T> entry, E extraData, class_332 graphics, float delta);
    
    void renderBase(EntryStack<T> entry, E extraData, class_332 graphics, class_4597.class_4598 immediate, Rectangle bounds, int mouseX, int mouseY, float delta);
    
    void afterBase(EntryStack<T> entry, E extraData, class_332 graphics, float delta);
    
    void renderOverlay(EntryStack<T> entry, E extraData, class_332 graphics, class_4597.class_4598 immediate, Rectangle bounds, int mouseX, int mouseY, float delta);
    
    /**
     * Ends the batch rendering, used to setup states, only called once with every batch.
     *
     * @param entry     the first entry in the batch
     * @param extraData the extra data returned from {@link #getExtraData(EntryStack)}
     * @param graphics  the graphics context
     * @param delta     the tick delta
     */
    void endBatch(EntryStack<T> entry, E extraData, class_332 graphics, float delta);
    
    @Override
    default void render(EntryStack<T> entry, class_332 graphics, Rectangle bounds, int mouseX, int mouseY, float delta) {
        class_4587 newStack = batchModifyMatrices(graphics.method_51448());
        graphics.method_51448().method_22903();
        graphics.method_51448().method_23760().method_23761().set(newStack.method_23760().method_23761());
        graphics.method_51448().method_23760().method_23762().set(newStack.method_23760().method_23762());
        E data = getExtraData(entry);
        startBatch(entry, data, graphics, delta);
        class_4597.class_4598 immediate = graphics.field_44658;
        renderBase(entry, data, graphics, immediate, bounds, mouseX, mouseY, delta);
        immediate.method_22993();
        renderOverlay(entry, data, graphics, immediate, bounds, mouseX, mouseY, delta);
        immediate.method_22993();
        endBatch(entry, data, graphics, delta);
        graphics.method_51448().method_22909();
    }
}
