/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.plugin.common.displays;

import com.mojang.serialization.codecs.RecordCodecBuilder;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.display.DisplaySerializer;
import me.shedaniel.rei.api.common.display.basic.BasicDisplay;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.entry.type.VanillaEntryTypes;
import me.shedaniel.rei.api.common.util.EntryIngredients;
import me.shedaniel.rei.api.common.util.EntryStacks;
import me.shedaniel.rei.plugin.common.BuiltinPlugin;
import me.shedaniel.rei.plugin.common.SmithingDisplay;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2960;
import net.minecraft.class_5455;
import net.minecraft.class_6880;
import net.minecraft.class_7225;
import net.minecraft.class_7924;
import net.minecraft.class_8053;
import net.minecraft.class_8054;
import net.minecraft.class_8055;
import net.minecraft.class_8056;
import net.minecraft.class_8057;
import net.minecraft.class_8060;
import net.minecraft.class_8062;
import net.minecraft.class_8786;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import net.minecraft.class_9334;
import net.minecraft.world.item.equipment.trim.*;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

public class DefaultSmithingDisplay extends BasicDisplay implements SmithingDisplay {
    public static final DisplaySerializer<DefaultSmithingDisplay> SERIALIZER = DisplaySerializer.of(
            RecordCodecBuilder.mapCodec(instance -> instance.group(
                    EntryIngredient.codec().listOf().fieldOf("inputs").forGetter(DefaultSmithingDisplay::getInputEntries),
                    EntryIngredient.codec().listOf().fieldOf("outputs").forGetter(DefaultSmithingDisplay::getOutputEntries),
                    SmithingRecipeType.CODEC.optionalFieldOf("type").forGetter(d -> d.type),
                    class_2960.field_25139.optionalFieldOf("location").forGetter(DefaultSmithingDisplay::getDisplayLocation)
            ).apply(instance, DefaultSmithingDisplay::new)),
            class_9139.method_56905(
                    EntryIngredient.streamCodec().method_56433(class_9135.method_56363()),
                    DefaultSmithingDisplay::getInputEntries,
                    EntryIngredient.streamCodec().method_56433(class_9135.method_56363()),
                    DefaultSmithingDisplay::getOutputEntries,
                    class_9135.method_56382(SmithingRecipeType.STREAM_CODEC),
                    d -> d.type,
                    class_9135.method_56382(class_2960.field_48267),
                    DefaultSmithingDisplay::getDisplayLocation,
                    DefaultSmithingDisplay::new
            ));
    
    private final Optional<SmithingRecipeType> type;
    
    @ApiStatus.Experimental
    public static DefaultSmithingDisplay ofTransforming(class_8786<class_8060> recipe) {
        return new DefaultSmithingDisplay(
                List.of(
                        recipe.comp_1933().method_64722().map(EntryIngredients::ofIngredient).orElse(EntryIngredient.empty()),
                        recipe.comp_1933().method_64723().map(EntryIngredients::ofIngredient).orElse(EntryIngredient.empty()),
                        recipe.comp_1933().method_64724().map(EntryIngredients::ofIngredient).orElse(EntryIngredient.empty())
                ),
                List.of(EntryIngredients.of(recipe.comp_1933().field_42033)),
                Optional.of(SmithingRecipeType.TRANSFORM),
                Optional.of(recipe.comp_1932().method_29177())
        );
    }
    
    public static List<DefaultSmithingDisplay> fromTrimming(class_8786<class_8062> recipe) {
        class_5455 registryAccess = BasicDisplay.registryAccess();
        List<DefaultSmithingDisplay> displays = new ArrayList<>();
        for (class_6880<class_1792> templateItem : (Iterable<class_6880<class_1792>>) recipe.comp_1933().method_64722().map(class_1856::method_8105).orElse(Stream.of())::iterator) {
            class_6880.class_6883<class_8056> trimPattern = getPatternFromTemplate(registryAccess, templateItem)
                    .orElse(null);
            if (trimPattern == null) continue;
            
            for (class_6880<class_1792> additionStack : (Iterable<class_6880<class_1792>>) recipe.comp_1933().method_64724().map(class_1856::method_8105).orElse(Stream.of())::iterator) {
                class_6880.class_6883<class_8054> trimMaterial = getMaterialFromIngredient(registryAccess, additionStack)
                        .orElse(null);
                if (trimMaterial == null) continue;
                
                EntryIngredient baseIngredient = recipe.comp_1933().method_64723().map(EntryIngredients::ofIngredient).orElse(EntryIngredient.empty());
                EntryIngredient templateOutput = baseIngredient.isEmpty() ? EntryIngredient.empty()
                        : getTrimmingOutput(registryAccess, EntryStacks.ofItemHolder(templateItem), baseIngredient.get(0), EntryStacks.ofItemHolder(additionStack));
                
                displays.add(new DefaultSmithingDisplay(List.of(
                        EntryIngredients.ofItemHolder(templateItem),
                        baseIngredient,
                        EntryIngredients.ofItemHolder(additionStack)
                ), List.of(templateOutput), Optional.of(SmithingRecipeType.TRIM), Optional.of(recipe.comp_1932().method_29177())));
            }
        }
        return displays;
    }
    
    public DefaultSmithingDisplay(List<EntryIngredient> inputs, List<EntryIngredient> outputs, Optional<class_2960> location) {
        this(inputs, outputs, Optional.empty(), location);
    }
    
    @ApiStatus.Experimental
    public DefaultSmithingDisplay(List<EntryIngredient> inputs, List<EntryIngredient> outputs, Optional<SmithingRecipeType> type, Optional<class_2960> location) {
        super(inputs, outputs, location);
        this.type = type;
    }
    
    @Override
    public CategoryIdentifier<?> getCategoryIdentifier() {
        return BuiltinPlugin.SMITHING;
    }
    
    @Override
    public DisplaySerializer<? extends Display> getSerializer() {
        return SERIALIZER;
    }
    
    @Nullable
    @Override
    public SmithingRecipeType type() {
        return type.orElse(null);
    }
    
    @ApiStatus.Experimental
    @ApiStatus.Internal
    public static EntryIngredient getTrimmingOutput(class_5455 registryAccess, EntryStack<?> template, EntryStack<?> base, EntryStack<?> addition) {
        if (template.getType() != VanillaEntryTypes.ITEM || base.getType() != VanillaEntryTypes.ITEM || addition.getType() != VanillaEntryTypes.ITEM) return EntryIngredient.empty();
        class_1799 templateItem = template.castValue();
        class_1799 baseItem = base.castValue();
        class_1799 additionItem = addition.castValue();
        class_6880.class_6883<class_8056> trimPattern = class_8057.method_48448(registryAccess, templateItem)
                .orElse(null);
        if (trimPattern == null) return EntryIngredient.empty();
        class_6880.class_6883<class_8054> trimMaterial = class_8055.method_48440(registryAccess, additionItem)
                .orElse(null);
        if (trimMaterial == null) return EntryIngredient.empty();
        class_8053 armorTrim = new class_8053(trimMaterial, trimPattern);
        class_8053 trim = baseItem.method_57824(class_9334.field_49607);
        if (trim != null && trim.method_48427(trimPattern, trimMaterial)) return EntryIngredient.empty();
        class_1799 newItem = baseItem.method_46651(1);
        newItem.method_57379(class_9334.field_49607, armorTrim);
        return EntryIngredients.of(newItem);
    }
    
    private static Optional<class_6880.class_6883<class_8056>> getPatternFromTemplate(class_7225.class_7874 provider, class_6880<class_1792> item) {
        return provider.method_46762(class_7924.field_42082)
                .method_42017()
                .filter(reference -> item == reference.comp_349().comp_1214())
                .findFirst();
    }
    
    private static Optional<class_6880.class_6883<class_8054>> getMaterialFromIngredient(class_7225.class_7874 provider, class_6880<class_1792> item) {
        return provider.method_46762(class_7924.field_42083)
                .method_42017()
                .filter(reference -> item == reference.comp_349().comp_1209())
                .findFirst();
    }
}
