/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.plugin.client;

import com.google.common.collect.*;
import com.google.gson.internal.LinkedTreeMap;
import dev.architectury.event.EventResult;
import dev.architectury.networking.NetworkManager;
import dev.architectury.platform.Platform;
import it.unimi.dsi.fastutil.objects.Reference2ObjectOpenHashMap;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.favorites.FavoriteEntry;
import me.shedaniel.rei.api.client.favorites.FavoriteEntryType;
import me.shedaniel.rei.api.client.plugins.REIClientPlugin;
import me.shedaniel.rei.api.client.registry.category.CategoryRegistry;
import me.shedaniel.rei.api.client.registry.display.DisplayRegistry;
import me.shedaniel.rei.api.client.registry.entry.CollapsibleEntryRegistry;
import me.shedaniel.rei.api.client.registry.entry.EntryRegistry;
import me.shedaniel.rei.api.client.registry.screen.ExclusionZones;
import me.shedaniel.rei.api.client.registry.screen.ScreenRegistry;
import me.shedaniel.rei.api.client.registry.transfer.TransferHandlerRegistry;
import me.shedaniel.rei.api.client.registry.transfer.simple.SimpleTransferHandler;
import me.shedaniel.rei.api.common.display.basic.BasicDisplay;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.entry.type.VanillaEntryTypes;
import me.shedaniel.rei.api.common.util.EntryIngredients;
import me.shedaniel.rei.api.common.util.EntryStacks;
import me.shedaniel.rei.impl.ClientInternals;
import me.shedaniel.rei.plugin.autocrafting.InventoryCraftingTransferHandler;
import me.shedaniel.rei.plugin.autocrafting.recipebook.DefaultRecipeBookHandler;
import me.shedaniel.rei.plugin.client.categories.*;
import me.shedaniel.rei.plugin.client.categories.anvil.DefaultAnvilCategory;
import me.shedaniel.rei.plugin.client.categories.beacon.DefaultBeaconBaseCategory;
import me.shedaniel.rei.plugin.client.categories.beacon.DefaultBeaconPaymentCategory;
import me.shedaniel.rei.plugin.client.categories.cooking.DefaultCookingCategory;
import me.shedaniel.rei.plugin.client.categories.crafting.DefaultCraftingCategory;
import me.shedaniel.rei.plugin.client.categories.tag.DefaultTagCategory;
import me.shedaniel.rei.plugin.client.displays.ClientsidedCookingDisplay;
import me.shedaniel.rei.plugin.client.displays.ClientsidedCraftingDisplay;
import me.shedaniel.rei.plugin.client.exclusionzones.DefaultPotionEffectExclusionZones;
import me.shedaniel.rei.plugin.client.exclusionzones.DefaultRecipeBookExclusionZones;
import me.shedaniel.rei.plugin.client.favorites.GameModeFavoriteEntry;
import me.shedaniel.rei.plugin.client.favorites.TimeFavoriteEntry;
import me.shedaniel.rei.plugin.client.favorites.WeatherFavoriteEntry;
import me.shedaniel.rei.plugin.common.BuiltinPlugin;
import me.shedaniel.rei.plugin.common.displays.*;
import me.shedaniel.rei.plugin.common.displays.anvil.AnvilRecipe;
import me.shedaniel.rei.plugin.common.displays.anvil.DefaultAnvilDisplay;
import me.shedaniel.rei.plugin.common.displays.beacon.DefaultBeaconBaseDisplay;
import me.shedaniel.rei.plugin.common.displays.beacon.DefaultBeaconPaymentDisplay;
import me.shedaniel.rei.plugin.common.displays.brewing.BrewingRecipe;
import me.shedaniel.rei.plugin.common.displays.brewing.DefaultBrewingDisplay;
import me.shedaniel.rei.plugin.common.displays.tag.DefaultTagDisplay;
import me.shedaniel.rei.plugin.common.displays.tag.TagNodes;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_10294;
import net.minecraft.class_10300;
import net.minecraft.class_10301;
import net.minecraft.class_1714;
import net.minecraft.class_1723;
import net.minecraft.class_1743;
import net.minecraft.class_1761;
import net.minecraft.class_1792;
import net.minecraft.class_1794;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1821;
import net.minecraft.class_1826;
import net.minecraft.class_1842;
import net.minecraft.class_1844;
import net.minecraft.class_1845;
import net.minecraft.class_1856;
import net.minecraft.class_1887;
import net.minecraft.class_1889;
import net.minecraft.class_1890;
import net.minecraft.class_1934;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3481;
import net.minecraft.class_3489;
import net.minecraft.class_3610;
import net.minecraft.class_3611;
import net.minecraft.class_3705;
import net.minecraft.class_3706;
import net.minecraft.class_3858;
import net.minecraft.class_3871;
import net.minecraft.class_3873;
import net.minecraft.class_3874;
import net.minecraft.class_3962;
import net.minecraft.class_472;
import net.minecraft.class_479;
import net.minecraft.class_481;
import net.minecraft.class_490;
import net.minecraft.class_518;
import net.minecraft.class_5953;
import net.minecraft.class_5955;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7225;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import net.minecraft.class_9334;
import net.minecraft.class_9886;
import net.minecraft.class_9890;
import net.minecraft.client.gui.screens.inventory.*;
import net.minecraft.world.inventory.*;
import net.minecraft.world.item.*;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.ApiStatus;

import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.IntStream;
import java.util.stream.Stream;

@Environment(EnvType.CLIENT)
@ApiStatus.Internal
public class DefaultClientPlugin implements REIClientPlugin, BuiltinClientPlugin {
    public DefaultClientPlugin() {
        ClientInternals.attachInstance((Supplier<Object>) () -> this, "builtinClientPlugin");
    }
    
    @Override
    public void registerBrewingRecipe(EntryIngredient input, EntryIngredient ingredient, EntryIngredient output) {
        DisplayRegistry.getInstance().add(new BrewingRecipe(input, ingredient, output));
    }
    
    @Override
    public void registerInformation(EntryIngredient ingredient, class_2561 name, UnaryOperator<List<class_2561>> textBuilder) {
        DisplayRegistry.getInstance().add(DefaultInformationDisplay.createFromEntries(ingredient, name).lines(textBuilder.apply(Lists.newArrayList())));
    }
    
    @Override
    public void registerEntries(EntryRegistry registry) {
        Multimap<class_1792, EntryStack<class_1799>> items = Multimaps.newListMultimap(new Reference2ObjectOpenHashMap<>()
                , ArrayList::new);
        
        for (Map.Entry<class_1761, Collection<class_1799>> entry : collectTabs().entrySet()) {
            try {
                for (class_1799 stack : entry.getValue()) {
                    try {
                        items.put(stack.method_7909(), EntryStacks.of(stack));
                    } catch (Exception ignore) {
                    }
                }
            } catch (Exception exception) {
                exception.printStackTrace();
            }
        }
        
        for (class_1792 item : class_7923.field_41178) {
            Collection<EntryStack<class_1799>> stacks = items.get(item);
            if (stacks.isEmpty()) {
                try {
                    registry.addEntry(EntryStacks.of(item.method_7854()));
                } catch (Exception ignore) {
                    registry.addEntry(EntryStacks.of(item));
                }
            } else {
                registry.addEntries(stacks);
            }
        }
        
        for (class_3611 fluid : class_7923.field_41173) {
            class_3610 state = fluid.method_15785();
            if (!state.method_15769() && state.method_15771()) {
                registry.addEntry(EntryStacks.of(fluid));
            }
        }
    }
    
    private static Map<class_1761, Collection<class_1799>> collectTabs() {
        try {
            return (Map<class_1761, Collection<class_1799>>) Class.forName(Platform.isForge() ? "me.shedaniel.rei.impl.client.forge.CreativeModeTabCollectorImpl"
                            : "me.shedaniel.rei.impl.client.fabric.CreativeModeTabCollectorImpl")
                    .getDeclaredMethod("collectTabs")
                    .invoke(null);
        } catch (ClassNotFoundException | NoSuchMethodException | InvocationTargetException | IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }
    
    @Override
    public void registerCollapsibleEntries(CollapsibleEntryRegistry registry) {
        registry.group(class_2960.method_60655("roughlyenoughitems", "enchanted_book"), class_2561.method_43471("item.minecraft.enchanted_book"),
                stack -> stack.getType() == VanillaEntryTypes.ITEM && stack.<class_1799>castValue().method_31574(class_1802.field_8598));
        registry.group(class_2960.method_60655("roughlyenoughitems", "potion"), class_2561.method_43471("item.minecraft.potion"),
                stack -> stack.getType() == VanillaEntryTypes.ITEM && stack.<class_1799>castValue().method_31574(class_1802.field_8574));
        registry.group(class_2960.method_60655("roughlyenoughitems", "splash_potion"), class_2561.method_43471("item.minecraft.splash_potion"),
                stack -> stack.getType() == VanillaEntryTypes.ITEM && stack.<class_1799>castValue().method_31574(class_1802.field_8436));
        registry.group(class_2960.method_60655("roughlyenoughitems", "lingering_potion"), class_2561.method_43471("item.minecraft.lingering_potion"),
                stack -> stack.getType() == VanillaEntryTypes.ITEM && stack.<class_1799>castValue().method_31574(class_1802.field_8150));
        registry.group(class_2960.method_60655("roughlyenoughitems", "spawn_egg"), class_2561.method_43471("text.rei.spawn_egg"),
                stack -> stack.getType() == VanillaEntryTypes.ITEM && stack.<class_1799>castValue().method_7909() instanceof class_1826);
        registry.group(class_2960.method_60655("roughlyenoughitems", "tipped_arrow"), class_2561.method_43471("item.minecraft.tipped_arrow"),
                stack -> stack.getType() == VanillaEntryTypes.ITEM && stack.<class_1799>castValue().method_31574(class_1802.field_8087));
        registry.group(class_2960.method_60655("roughlyenoughitems", "music_disc"), class_2561.method_43471("text.rei.music_disc"),
                stack -> stack.getType() == VanillaEntryTypes.ITEM && stack.<class_1799>castValue().method_57826(class_9334.field_52175));
    }
    
    @Override
    public void registerCategories(CategoryRegistry registry) {
        registry.add(
                new DefaultCraftingCategory(),
                new DefaultCookingCategory(SMELTING, EntryStacks.of(class_1802.field_8732), "category.rei.smelting", 200),
                new DefaultCookingCategory(SMOKING, EntryStacks.of(class_1802.field_16309), "category.rei.smoking", 100),
                new DefaultCookingCategory(BLASTING, EntryStacks.of(class_1802.field_16306), "category.rei.blasting", 100),
                new DefaultCampfireCategory(),
                new DefaultStoneCuttingCategory(),
                new DefaultFuelCategory(),
                new DefaultBrewingCategory(),
                new DefaultCompostingCategory(),
                new DefaultStrippingCategory(),
                new DefaultSmithingCategory(),
                new DefaultAnvilCategory(),
                new DefaultBeaconBaseCategory(),
                new DefaultBeaconPaymentCategory(),
                new DefaultTillingCategory(),
                new DefaultPathingCategory(),
                new DefaultWaxingCategory(),
                new DefaultWaxScrapingCategory(),
                new DefaultOxidizingCategory(),
                new DefaultOxidationScrapingCategory()
        );
        
        registry.addWorkstations(CRAFTING, EntryStacks.of(class_1802.field_8465));
        registry.addWorkstations(SMELTING, EntryStacks.of(class_1802.field_8732));
        registry.addWorkstations(SMOKING, EntryStacks.of(class_1802.field_16309));
        registry.addWorkstations(BLASTING, EntryStacks.of(class_1802.field_16306));
        registry.addWorkstations(CAMPFIRE, EntryStacks.of(class_1802.field_17346), EntryStacks.of(class_1802.field_23842));
        registry.addWorkstations(FUEL, EntryStacks.of(class_1802.field_8732), EntryStacks.of(class_1802.field_16309), EntryStacks.of(class_1802.field_16306));
        registry.addWorkstations(BREWING, EntryStacks.of(class_1802.field_8740));
        registry.addWorkstations(ANVIL, EntryStacks.of(class_1802.field_8782));
        registry.addWorkstations(STONE_CUTTING, EntryStacks.of(class_1802.field_16305));
        registry.addWorkstations(COMPOSTING, EntryStacks.of(class_1802.field_17530));
        registry.addWorkstations(SMITHING, EntryStacks.of(class_1802.field_16308));
        registry.addWorkstations(BEACON_BASE, EntryStacks.of(class_1802.field_8668));
        registry.addWorkstations(BEACON_PAYMENT, EntryStacks.of(class_1802.field_8668));
        registry.addWorkstations(WAXING, EntryStacks.of(class_1802.field_20414));
        
        registry.configure(INFO, config -> config.setQuickCraftingEnabledByDefault(false));
        registry.configure(TAG, config -> config.setQuickCraftingEnabledByDefault(false));
        
        registry.registerVisibilityPredicate(category -> {
            if (category instanceof DefaultTagCategory && class_310.method_1551().method_1576() == null && !NetworkManager.canServerReceive(TagNodes.REQUEST_TAGS_PACKET_C2S)) {
                return EventResult.interruptFalse();
            }
            
            return EventResult.pass();
        });
        
        Set<class_1792> axes = Sets.newHashSet(), hoes = Sets.newHashSet(), shovels = Sets.newHashSet();
        EntryRegistry.getInstance().getEntryStacks().filter(stack -> stack.getValueType() == class_1799.class).map(stack -> ((class_1799) stack.getValue()).method_7909()).forEach(item -> {
            if (item instanceof class_1743 && axes.add(item)) {
                registry.addWorkstations(STRIPPING, EntryStacks.of(item));
                registry.addWorkstations(WAX_SCRAPING, EntryStacks.of(item));
                registry.addWorkstations(OXIDATION_SCRAPING, EntryStacks.of(item));
            }
            if (item instanceof class_1794 && hoes.add(item)) {
                registry.addWorkstations(TILLING, EntryStacks.of(item));
            }
            if (item instanceof class_1821 && shovels.add(item)) {
                registry.addWorkstations(PATHING, EntryStacks.of(item));
            }
        });
        for (EntryStack<?> stack : getTag(class_2960.method_60655("c", "axes"))) {
            if (axes.add(stack.<class_1799>castValue().method_7909())) {
                registry.addWorkstations(STRIPPING, stack);
                registry.addWorkstations(WAX_SCRAPING, stack);
                registry.addWorkstations(OXIDATION_SCRAPING, stack);
            }
        }
        for (EntryStack<?> stack : getTag(class_2960.method_60655("c", "hoes"))) {
            if (hoes.add(stack.<class_1799>castValue().method_7909())) registry.addWorkstations(TILLING, stack);
        }
        for (EntryStack<?> stack : getTag(class_2960.method_60655("c", "shovels"))) {
            if (shovels.add(stack.<class_1799>castValue().method_7909())) registry.addWorkstations(PATHING, stack);
        }
    }
    
    private static EntryIngredient getTag(class_2960 tagId) {
        return EntryIngredients.ofItemTag(class_6862.method_40092(class_7924.field_41197, tagId));
    }
    
    @Override
    public void registerDisplays(DisplayRegistry registry) {
        CategoryRegistry.getInstance().add(new DefaultInformationCategory(), new DefaultTagCategory());
        
        registry.beginRecipeFiller(class_10300.class)
                .filterType(class_10300.field_54667)
                .fill(ClientsidedCraftingDisplay.Shaped::new);
        registry.beginRecipeFiller(class_10301.class)
                .filterType(class_10301.field_54670)
                .fill(ClientsidedCraftingDisplay.Shapeless::new);
        registry.beginRecipeFiller(class_10294.class)
                .filterType(class_10294.field_54660)
                .filter((display, r) -> EntryIngredients.ofSlotDisplay(display.comp_3259()).contains(EntryStacks.of(class_1802.field_8732)))
                .fill(ClientsidedCookingDisplay.Smelting::new);
        registry.beginRecipeFiller(class_10294.class)
                .filterType(class_10294.field_54660)
                .filter((display, r) -> EntryIngredients.ofSlotDisplay(display.comp_3259()).contains(EntryStacks.of(class_1802.field_16309)))
                .fill(ClientsidedCookingDisplay.Smoking::new);
        registry.beginRecipeFiller(class_10294.class)
                .filterType(class_10294.field_54660)
                .filter((display, r) -> EntryIngredients.ofSlotDisplay(display.comp_3259()).contains(EntryStacks.of(class_1802.field_16306)))
                .fill(ClientsidedCookingDisplay.Blasting::new);
        registry.beginFiller(AnvilRecipe.class)
                .fill(DefaultAnvilDisplay::new);
        registry.beginFiller(BrewingRecipe.class)
                .fill(DefaultBrewingDisplay::new);
        registry.beginFiller(class_6862.class)
                .fill(tagKey -> {
                    if (tagKey.method_41007(class_7924.field_41197)) {
                        return DefaultTagDisplay.ofItems(tagKey);
                    } else if (tagKey.method_41007(class_7924.field_41254)) {
                        return DefaultTagDisplay.ofItems(tagKey);
                    } else if (tagKey.method_41007(class_7924.field_41270)) {
                        return DefaultTagDisplay.ofFluids(tagKey);
                    }
                    
                    return null;
                });
        // TODO: Fuel
//        for (Map.Entry<Item, Integer> entry : AbstractFurnaceBlockEntity.getFuel().entrySet()) {
//            registry.add(new DefaultFuelDisplay(Collections.singletonList(EntryIngredients.of(entry.getKey())), Collections.emptyList(), entry.getValue()));
//        }
        if (class_3962.field_17566.isEmpty()) {
            class_3962.method_17758();
        }
        Iterator<List<EntryIngredient>> iterator = Iterators.partition(class_3962.field_17566.object2FloatEntrySet().stream().sorted(Map.Entry.comparingByValue()).map(entry -> EntryIngredients.of(entry.getKey())).iterator(), 35);
        while (iterator.hasNext()) {
            List<EntryIngredient> entries = iterator.next();
            registry.add(new DefaultCompostingDisplay(entries, Collections.singletonList(EntryIngredients.of(new class_1799(class_1802.field_8324)))));
        }
        DummyAxeItem.getStrippedBlocksMap().entrySet().stream().sorted(Comparator.comparing(b -> class_7923.field_41175.method_10221(b.getKey()))).forEach(set -> {
            registry.add(new DefaultStrippingDisplay(EntryStacks.of(set.getKey()), EntryStacks.of(set.getValue())));
        });
        DummyShovelItem.getPathBlocksMap().entrySet().stream().sorted(Comparator.comparing(b -> class_7923.field_41175.method_10221(b.getKey()))).forEach(set -> {
            registry.add(new DefaultPathingDisplay(EntryStacks.of(set.getKey()), EntryStacks.of(set.getValue().method_26204())));
        });
        registry.add(new DefaultBeaconBaseDisplay(Collections.singletonList(EntryIngredients.ofItemTag(class_3481.field_22275)), Collections.emptyList()));
        registry.add(new DefaultBeaconPaymentDisplay(Collections.singletonList(EntryIngredients.ofItemTag(class_3489.field_22277)), Collections.emptyList()));
        class_5953.field_29560.get().entrySet().stream().sorted(Comparator.comparing(b -> class_7923.field_41175.method_10221(b.getKey()))).forEach(set -> {
            registry.add(new DefaultWaxingDisplay(EntryStacks.of(set.getKey()), EntryStacks.of(set.getValue())));
        });
        class_5953.field_29561.get().entrySet().stream().sorted(Comparator.comparing(b -> class_7923.field_41175.method_10221(b.getKey()))).forEach(set -> {
            registry.add(new DefaultWaxScrapingDisplay(EntryStacks.of(set.getKey()), EntryStacks.of(set.getValue())));
        });
        class_5955.field_29564.get().entrySet().stream().sorted(Comparator.comparing(b -> class_7923.field_41175.method_10221(b.getKey()))).forEach(set -> {
            registry.add(new DefaultOxidizingDisplay(EntryStacks.of(set.getKey()), EntryStacks.of(set.getValue())));
        });
        class_5955.field_29565.get().entrySet().stream().sorted(Comparator.comparing(b -> class_7923.field_41175.method_10221(b.getKey()))).forEach(set -> {
            registry.add(new DefaultOxidationScrapingDisplay(EntryStacks.of(set.getKey()), EntryStacks.of(set.getValue())));
        });
        if (Platform.isFabric()) {
            Set<class_6880<class_1842>> potions = Collections.newSetFromMap(new LinkedTreeMap<>(Comparator.comparing(class_6880::getRegisteredName), false));
            class_1845 brewing = class_310.method_1551().field_1687.method_59547();
            for (class_1856 container : brewing.field_51403) {
                for (class_1845.class_1846<class_1842> mix : brewing.field_51404) {
                    class_6880<class_1842> from = mix.comp_2190();
                    class_1856 ingredient = mix.comp_2191();
                    class_6880<class_1842> to = mix.comp_2192();
                    EntryIngredient base = EntryIngredients.ofIngredient(container)
                            .map(stack -> {
                                EntryStack<?> copied = stack.copy();
                                copied.<class_1799>castValue().method_57379(class_9334.field_49651, new class_1844(from));
                                return copied;
                            });
                    EntryIngredient output = EntryIngredients.ofIngredient(container)
                            .map(stack -> {
                                EntryStack<?> copied = stack.copy();
                                copied.<class_1799>castValue().method_57379(class_9334.field_49651, new class_1844(to));
                                return copied;
                            });
                    registerBrewingRecipe(base, EntryIngredients.ofIngredient(ingredient), output);
                    potions.add(from);
                    potions.add(to);
                }
            }
            for (class_6880<class_1842> potion : potions) {
                for (class_1845.class_1846<class_1792> mix : brewing.field_51405) {
                    class_6880<class_1792> from = mix.comp_2190();
                    class_1856 ingredient = mix.comp_2191();
                    class_6880<class_1792> to = mix.comp_2192();
                    class_1799 baseStack = new class_1799(from);
                    baseStack.method_57379(class_9334.field_49651, new class_1844(potion));
                    EntryIngredient base = EntryIngredients.of(baseStack);
                    class_1799 output = new class_1799(to);
                    output.method_57379(class_9334.field_49651, new class_1844(potion));
                    registerBrewingRecipe(base, EntryIngredients.ofIngredient(ingredient), EntryIngredients.of(output));
                }
            }
        } else {
            registerForgePotions(registry, this);
        }
        
        for (class_1792 item : class_7923.field_41178) {
            class_1799 stack = item.method_7854();
            if (!stack.method_7963()) continue;
            if (item.method_57347().method_57832(class_9334.field_53696)) {
                class_9890 repairable = item.method_57347().method_57829(class_9334.field_53696);
                EntryIngredient repairMaterialBase = EntryIngredients.ofItemsHolderSet(repairable.comp_2939());
                if (repairMaterialBase.isEmpty()) continue;
                
                for (int[] i = {1}; i[0] <= 4; i[0]++) {
                    class_1799 baseStack = item.method_7854();
                    int toRepair = i[0] == 4 ? baseStack.method_7936() : baseStack.method_7936() / 4 * i[0];
                    baseStack.method_7974(toRepair);
                    EntryIngredient repairMaterial = repairMaterialBase.map(s -> {
                        EntryStack<?> newStack = s.copy();
                        newStack.<class_1799>castValue().method_7939(i[0]);
                        return newStack;
                    });
                    Optional<Pair<class_1799, Integer>> output = DefaultAnvilDisplay.calculateOutput(baseStack, repairMaterial.get(0).castValue());
                    if (output.isEmpty()) continue;
                    registry.add(new DefaultAnvilDisplay(List.of(EntryIngredients.of(baseStack), repairMaterial),
                            Collections.singletonList(EntryIngredients.of(output.get().getLeft())), Optional.empty(), OptionalInt.of(output.get().getRight())));
                }
            }
        }
        List<Pair<class_1889, class_1799>> enchantmentBooks = BasicDisplay.registryAccess().method_46759(class_7924.field_41265)
                .stream()
                .flatMap(class_7225::method_42017)
                .flatMap(holder -> {
                    if (!holder.method_40227()) return Stream.empty();
                    class_1887 enchantment = holder.comp_349();
                    if (enchantment.method_8183() - enchantment.method_8187() >= 10) {
                        return IntStream.of(enchantment.method_8187(), enchantment.method_8183())
                                .mapToObj(lvl -> new class_1889(holder, lvl));
                    } else {
                        return IntStream.rangeClosed(enchantment.method_8187(), enchantment.method_8183())
                                .mapToObj(lvl -> new class_1889(holder, lvl));
                    }
                })
                .map(instance -> {
                    return Pair.of(instance, class_1890.method_61711(instance));
                })
                .toList();
        EntryRegistry.getInstance().getEntryStacks().forEach(stack -> {
            if (stack.getType() != VanillaEntryTypes.ITEM) return;
            class_1799 itemStack = stack.castValue();
            if (!itemStack.method_7923()) return;
            for (Pair<class_1889, class_1799> pair : enchantmentBooks) {
                if (!pair.getKey().field_9093.comp_349().method_8192(itemStack)) continue;
                Optional<Pair<class_1799, Integer>> output = DefaultAnvilDisplay.calculateOutput(itemStack, pair.getValue());
                if (output.isEmpty()) continue;
                registry.add(new DefaultAnvilDisplay(List.of(EntryIngredients.of(itemStack), EntryIngredients.of(pair.getValue())),
                        Collections.singletonList(EntryIngredients.of(output.get().getLeft())), Optional.empty(), OptionalInt.of(output.get().getRight())));
            }
        });
        
        for (class_2378<?> reg : class_7923.field_41167) {
            reg.method_40272().forEach(tagPair -> tagPair.method_40248().ifLeft(registry::add));
        }
    }
    
    protected void registerForgePotions(DisplayRegistry registry, BuiltinClientPlugin clientPlugin) {
        
    }
    
    @Override
    public void registerExclusionZones(ExclusionZones zones) {
        zones.register(class_490.class, new DefaultPotionEffectExclusionZones<>());
        zones.register(class_481.class, new DefaultPotionEffectExclusionZones<>());
        zones.register(class_518.class, new DefaultRecipeBookExclusionZones());
    }
    
    @Override
    public void registerScreens(ScreenRegistry registry) {
        registry.registerContainerClickArea(new Rectangle(88, 32, 28, 23), class_479.class, CRAFTING);
        registry.registerContainerClickArea(new Rectangle(137, 29, 10, 13), class_490.class, CRAFTING);
        registry.registerContainerClickArea(new Rectangle(97, 16, 14, 30), class_472.class, BREWING);
        registry.registerContainerClickArea(new Rectangle(78, 32, 28, 23), class_3873.class, SMELTING);
        registry.registerContainerClickArea(new Rectangle(78, 32, 28, 23), class_3874.class, SMOKING);
        registry.registerContainerClickArea(new Rectangle(78, 32, 28, 23), class_3871.class, BLASTING);
    }
    
    @Override
    public void registerTransferHandlers(TransferHandlerRegistry registry) {
        registry.register(SimpleTransferHandler.create(class_1714.class, BuiltinPlugin.CRAFTING,
                new SimpleTransferHandler.IntRange(1, 10)));
        registry.register(new InventoryCraftingTransferHandler(SimpleTransferHandler.create(class_1723.class, BuiltinPlugin.CRAFTING,
                new SimpleTransferHandler.IntRange(1, 5))));
        registry.register(SimpleTransferHandler.create(class_3858.class, BuiltinPlugin.SMELTING,
                new SimpleTransferHandler.IntRange(0, 1)));
        registry.register(SimpleTransferHandler.create(class_3706.class, BuiltinPlugin.SMOKING,
                new SimpleTransferHandler.IntRange(0, 1)));
        registry.register(SimpleTransferHandler.create(class_3705.class, BuiltinPlugin.BLASTING,
                new SimpleTransferHandler.IntRange(0, 1)));
        registry.register(new DefaultRecipeBookHandler());
    }
    
    @Override
    public void registerFavorites(FavoriteEntryType.Registry registry) {
        registry.register(GameModeFavoriteEntry.ID, GameModeFavoriteEntry.Type.INSTANCE);
        registry.getOrCrateSection(class_2561.method_43471(GameModeFavoriteEntry.TRANSLATION_KEY))
                .add(Stream.concat(
                        Arrays.stream(class_1934.values())
                                .filter(type -> type.method_8379() >= 0),
                        Stream.of((class_1934) null)
                ).<FavoriteEntry>map(GameModeFavoriteEntry::new).toArray(FavoriteEntry[]::new));
        registry.register(WeatherFavoriteEntry.ID, WeatherFavoriteEntry.Type.INSTANCE);
        registry.getOrCrateSection(class_2561.method_43471(WeatherFavoriteEntry.TRANSLATION_KEY))
                .add(Stream.concat(
                        Arrays.stream(WeatherFavoriteEntry.Weather.values()),
                        Stream.of((WeatherFavoriteEntry.Weather) null)
                ).<FavoriteEntry>map(WeatherFavoriteEntry::new).toArray(FavoriteEntry[]::new));
        registry.register(TimeFavoriteEntry.ID, TimeFavoriteEntry.Type.INSTANCE);
        registry.getOrCrateSection(class_2561.method_43471(TimeFavoriteEntry.TRANSLATION_KEY))
                .add(Stream.concat(
                        Arrays.stream(TimeFavoriteEntry.Time.values()),
                        Stream.of((TimeFavoriteEntry.Time) null)
                ).<FavoriteEntry>map(TimeFavoriteEntry::new).toArray(FavoriteEntry[]::new));
    }
    
    @Override
    public double getPriority() {
        return -100;
    }
    
    public static class DummyShovelItem extends class_1821 {
        public DummyShovelItem(class_9886 material, float damage, float speed, class_1793 properties) {
            super(material, damage, speed, properties);
        }
        
        public static Map<class_2248, class_2680> getPathBlocksMap() {
            return field_8912;
        }
    }
    
    public static class DummyAxeItem extends class_1743 {
        public DummyAxeItem(class_9886 material, float damage, float speed, class_1793 properties) {
            super(material, damage, speed, properties);
        }
        
        public static Map<class_2248, class_2248> getStrippedBlocksMap() {
            return field_7898;
        }
    }
}
