/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.common.transfer;

import me.shedaniel.rei.api.common.entry.InputIngredient;
import me.shedaniel.rei.api.common.transfer.ItemRecipeFinder;
import me.shedaniel.rei.api.common.transfer.info.stack.SlotAccessor;
import net.minecraft.class_1263;
import net.minecraft.class_1703;
import net.minecraft.class_1799;
import net.minecraft.class_3222;
import net.minecraft.class_9334;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public abstract class InputSlotCrafter<T extends class_1703, C extends class_1263> {
    protected T container;
    private Iterable<SlotAccessor> inputStacks;
    private Iterable<SlotAccessor> inventoryStacks;
    protected class_3222 player;
    
    protected InputSlotCrafter(T container) {
        this.container = container;
    }
    
    public void fillInputSlots(class_3222 player, boolean hasShift) {
        this.player = player;
        this.inventoryStacks = this.getInventorySlots();
        this.inputStacks = this.getInputSlots();
        
        // Return the already placed items on the grid
        this.cleanInputs();
        
        ItemRecipeFinder recipeFinder = new ItemRecipeFinder();
        this.populateRecipeFinder(recipeFinder);
        List<List<class_1799>> ingredients = new ArrayList<>();
        for (InputIngredient<class_1799> itemStacks : this.getInputs()) {
            ingredients.add(itemStacks.get());
        }
        
        if (recipeFinder.findRecipe(ingredients, 1, null)) {
            this.fillInputSlots(recipeFinder, ingredients, hasShift);
        } else {
            this.cleanInputs();
            this.markDirty();
            throw new NotEnoughMaterialsException();
        }
        
        this.markDirty();
    }
    
    protected abstract Iterable<SlotAccessor> getInputSlots();
    
    protected abstract Iterable<SlotAccessor> getInventorySlots();
    
    protected abstract List<InputIngredient<class_1799>> getInputs();
    
    protected abstract void populateRecipeFinder(ItemRecipeFinder recipeFinder);
    
    protected abstract void markDirty();
    
    public void alignRecipeToGrid(Iterable<SlotAccessor> inputStacks, Iterator<class_1799> recipeItems, int craftsAmount) {
        for (SlotAccessor inputStack : inputStacks) {
            if (!recipeItems.hasNext()) {
                return;
            }
            
            this.acceptAlignedInput(recipeItems.next(), inputStack, craftsAmount);
        }
    }
    
    public void acceptAlignedInput(class_1799 toBeTakenStack, SlotAccessor inputStack, int craftsAmount) {
        if (!toBeTakenStack.method_7960()) {
            for (int i = 0; i < craftsAmount; ++i) {
                this.fillInputSlot(inputStack, toBeTakenStack);
            }
        }
    }
    
    protected void fillInputSlot(SlotAccessor slot, class_1799 toBeTakenStack) {
        SlotAccessor takenSlot = this.takeInventoryStack(toBeTakenStack);
        if (takenSlot != null) {
            class_1799 takenStack = takenSlot.getItemStack().method_7972();
            if (!takenStack.method_7960()) {
                if (takenStack.method_7947() > 1) {
                    takenSlot.takeStack(1);
                } else {
                    takenSlot.setItemStack(class_1799.field_8037);
                }
                
                takenStack.method_7939(1);
                if (!slot.canPlace(takenStack)) {
                    return;
                }
                
                if (slot.getItemStack().method_7960()) {
                    slot.setItemStack(takenStack);
                } else {
                    slot.getItemStack().method_7933(1);
                }
            }
        }
    }
    
    protected void fillInputSlots(ItemRecipeFinder recipeFinder, List<List<class_1799>> ingredients, boolean hasShift) {
        int recipeCrafts = recipeFinder.countRecipeCrafts(ingredients, Integer.MAX_VALUE, null);
        int amountToFill = hasShift ? recipeCrafts : 1;
        List<class_1799> recipeItems = new ArrayList<>();
        if (recipeFinder.findRecipe(ingredients, amountToFill, recipeItems::add)) {
            int finalCraftsAmount = amountToFill;
            
            for (class_1799 itemId : recipeItems) {
                // Fix issue with empty item id (grid slot) [shift-click issue]
                if (itemId.method_7960()) {
                    continue;
                }
                finalCraftsAmount = Math.min(finalCraftsAmount, itemId.method_7914());
            }
            
            recipeItems.clear();
            
            if (recipeFinder.findRecipe(ingredients, finalCraftsAmount, recipeItems::add)) {
                this.cleanInputs();
                this.alignRecipeToGrid(inputStacks, recipeItems.iterator(), finalCraftsAmount);
            }
        }
    }
    
    protected abstract void cleanInputs();
    
    @Nullable
    public SlotAccessor takeInventoryStack(class_1799 itemStack) {
        boolean rejectedModification = false;
        for (SlotAccessor inventoryStack : inventoryStacks) {
            class_1799 itemStack1 = inventoryStack.getItemStack();
            if (!itemStack1.method_7960() && areItemsEqual(itemStack, itemStack1) && !itemStack1.method_7986() && !itemStack1.method_7942() && !itemStack1.method_57826(class_9334.field_49631)) {
                if (!inventoryStack.allowModification(player)) {
                    rejectedModification = true;
                } else {
                    return inventoryStack;
                }
            }
        }
        
        if (rejectedModification) {
            throw new IllegalStateException("Unable to take item from inventory due to slot not allowing modification! Item requested: " + itemStack);
        }
        
        return null;
    }
    
    private static boolean areItemsEqual(class_1799 stack1, class_1799 stack2) {
        return class_1799.method_31577(stack1, stack2);
    }
    
    public static class NotEnoughMaterialsException extends RuntimeException {
    }
}
