/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.widget.favorites.history;

import com.google.common.collect.Iterables;
import com.mojang.serialization.DataResult;
import it.unimi.dsi.fastutil.objects.Reference2ObjectLinkedOpenHashMap;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.registry.category.CategoryRegistry;
import me.shedaniel.rei.api.common.category.CategoryIdentifier;
import me.shedaniel.rei.api.common.display.Display;
import me.shedaniel.rei.api.common.display.basic.BasicDisplay;
import me.shedaniel.rei.api.common.plugins.PluginManager;
import me.shedaniel.rei.impl.client.config.ConfigManagerImpl;
import me.shedaniel.rei.impl.common.InternalLogger;
import me.shedaniel.rei.impl.common.registry.displays.DisplayKey;
import me.shedaniel.rei.impl.common.registry.displays.DisplaysHolder;
import net.minecraft.class_156;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2960;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public class DisplayHistoryManager {
    public static final DisplayHistoryManager INSTANCE = new DisplayHistoryManager();
    private Map<String, DisplayEntry> entries = new LinkedHashMap<>();
    private final Map<Display, DisplayEntry> displayToEntries = new Reference2ObjectLinkedOpenHashMap<>();
    private long lastCheckTime = -1;
    
    @Nullable
    public Object getPossibleOrigin(DisplaysHolder.ByKey holder, Display display) {
        DisplayEntry entry = displayToEntries.get(display);
        if (entry == null) return null;
        Optional<class_2960> location = display.getDisplayLocation();
        if (location.isEmpty()) return null;
        Set<Display> displays = holder.getDisplaysByKey(DisplayKey.create(display.getCategoryIdentifier(), location.get()));
        for (Display d : displays) {
            if (!displayToEntries.containsKey(d)) {
                Object origin = holder.getDisplayOrigin(d);
                
                if (origin != null) {
                    return origin;
                }
            }
        }
        
        return null;
    }
    
    public Map<Display, DisplayEntry> getDisplayToEntries() {
        return displayToEntries;
    }
    
    public Collection<DisplayEntry> getEntries(DisplayHistoryWidget parent) {
        if ((lastCheckTime == -1 || class_156.method_658() - lastCheckTime > 4000) && !PluginManager.areAnyReloading()) {
            updateEntries(parent);
            lastCheckTime = class_156.method_658();
        }
        
        return Collections.unmodifiableCollection(entries.values());
    }
    
    private void updateEntries(DisplayHistoryWidget parent) {
        List<class_2487> displayHistory = ConfigManagerImpl.getInstance().getConfig().getDisplayHistory();
        Map<String, DisplayEntry> copy = new LinkedHashMap<>(entries);
        entries.clear();
        displayToEntries.clear();
        for (class_2487 tag : displayHistory) {
            String uuid = tag.method_10558("DisplayHistoryUUID");
            
            DisplayEntry entry = copy.get(uuid);
            if (entry != null) {
                entries.put(entry.getUuid().toString(), entry);
                displayToEntries.put(entry.getDisplay(), entry);
            } else if (tag.method_10577("DisplayHistoryContains")) {
                try {
                    CategoryIdentifier<?> categoryIdentifier = CategoryIdentifier.of(tag.method_10558("DisplayHistoryCategory"));
                    if (CategoryRegistry.getInstance().tryGet(categoryIdentifier).isPresent()) {
                        DataResult<Display> result = Display.codec().parse(BasicDisplay.registryAccess().method_57093(class_2509.field_11560), tag.method_10562("DisplayHistoryData"));
                        Display display = result.getOrThrow();
                        DisplayEntry newEntry = new DisplayEntry(parent, display, null);
                        newEntry.setUuid(UUID.fromString(uuid));
                        entries.put(newEntry.getUuid().toString(), newEntry);
                        displayToEntries.put(newEntry.getDisplay(), newEntry);
                    }
                } catch (Exception e) {
                    InternalLogger.getInstance().warn("Failed to read display history entry", e);
                }
            }
        }
    }
    
    public void removeEntry(DisplayEntry entry) {
        this.entries.remove(entry.getUuid().toString());
        this.displayToEntries.remove(entry.getDisplay());
        List<class_2487> displayHistory = ConfigManagerImpl.getInstance().getConfig().getDisplayHistory();
        displayHistory.removeIf(tag -> tag.method_10558("DisplayHistoryUUID").equals(entry.getUuid().toString()));
        save();
    }
    
    public void addEntry(DisplayHistoryWidget parent, @Nullable Rectangle bounds, Display display) {
        List<class_2487> displayHistory = ConfigManagerImpl.getInstance().getConfig().getDisplayHistory();
        Iterator<DisplayEntry> iterator = this.entries.values().iterator();
        while (iterator.hasNext()) {
            DisplayEntry entry = iterator.next();
            if (entry.getDisplay() == display) {
                displayHistory.removeIf(tag -> tag.method_10558("DisplayHistoryUUID").equals(entry.getUuid().toString()));
                this.displayToEntries.remove(entry.getDisplay());
                iterator.remove();
            }
        }
        DisplayEntry newEntry = new DisplayEntry(parent, display, bounds);
        Map<String, DisplayEntry> copy = new LinkedHashMap<>();
        copy.put(newEntry.getUuid().toString(), newEntry);
        copy.putAll(this.entries);
        this.entries = copy;
        this.displayToEntries.clear();
        for (DisplayEntry entry : this.entries.values()) {
            this.displayToEntries.put(entry.getDisplay(), entry);
        }
        while (entries.size() >= 10) {
            DisplayEntry entry = Iterables.get(entries.values(), entries.size() - 1);
            displayHistory.removeIf(tag -> tag.method_10558("DisplayHistoryUUID").equals(entry.getUuid().toString()));
            this.entries.remove(entry.getUuid().toString());
            this.displayToEntries.remove(entry.getDisplay());
        }
        
        class_2487 compoundTag = new class_2487();
        compoundTag.method_10556("DisplayHistoryContains", false);
        compoundTag.method_10582("DisplayHistoryUUID", newEntry.getUuid().toString());
        compoundTag.method_10582("DisplayHistoryCategory", display.getCategoryIdentifier().toString());
        displayHistory.add(0, compoundTag);
        
        save();
    }
    
    private void save() {
        List<class_2487> displayHistory = ConfigManagerImpl.getInstance().getConfig().getDisplayHistory();
        for (class_2487 compoundTag : displayHistory) {
            String uuid = compoundTag.method_10558("DisplayHistoryUUID");
            DisplayEntry entry = entries.get(uuid);
            
            if (entry != null) {
                compoundTag.method_10556("DisplayHistoryContains", false);
                Display display = entry.getDisplay();
                
                if (display.getSerializer() != null) {
                    try {
                        DataResult<class_2520> displayTag = Display.codec().encodeStart(BasicDisplay.registryAccess().method_57093(class_2509.field_11560), display);
                        compoundTag.method_10566("DisplayHistoryData", displayTag.getOrThrow());
                        compoundTag.method_10556("DisplayHistoryContains", true);
                    } catch (Exception e) {
                        InternalLogger.getInstance().warn("Failed to save display history entry", e);
                    }
                }
            }
        }
        
        ConfigManagerImpl.getInstance().saveConfig();
    }
}
