/*
 * This file is licensed under the MIT License, part of Roughly Enough Items.
 * Copyright (c) 2018, 2019, 2020, 2021, 2022, 2023 shedaniel
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package me.shedaniel.rei.impl.client.gui.widget;

import com.mojang.blaze3d.systems.RenderSystem;
import dev.architectury.registry.ReloadListenerRegistry;
import it.unimi.dsi.fastutil.longs.Long2LongMap;
import it.unimi.dsi.fastutil.longs.Long2LongOpenHashMap;
import me.shedaniel.clothconfig2.api.LazyResettable;
import me.shedaniel.math.Rectangle;
import me.shedaniel.rei.api.client.config.ConfigObject;
import me.shedaniel.rei.api.client.entry.renderer.EntryRenderer;
import me.shedaniel.rei.api.client.registry.entry.EntryRegistry;
import me.shedaniel.rei.api.common.entry.EntryStack;
import me.shedaniel.rei.api.common.entry.type.VanillaEntryTypes;
import me.shedaniel.rei.api.common.util.EntryStacks;
import me.shedaniel.rei.impl.common.InternalLogger;
import net.minecraft.class_1011;
import net.minecraft.class_10142;
import net.minecraft.class_10366;
import net.minecraft.class_1043;
import net.minecraft.class_1921;
import net.minecraft.class_290;
import net.minecraft.class_293;
import net.minecraft.class_2960;
import net.minecraft.class_308;
import net.minecraft.class_310;
import net.minecraft.class_3264;
import net.minecraft.class_332;
import net.minecraft.class_3902;
import net.minecraft.class_4668;
import net.minecraft.class_6367;
import net.minecraft.class_9851;
import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4f;
import org.joml.Matrix4fStack;

import java.util.List;

public class CachedEntryListRender {
    public static final int RESOLUTION = 64;
    public static class_1043 cachedTexture;
    public static class_2960 cachedTextureLocation;
    public static Long2LongMap hash = new Long2LongOpenHashMap();
    public static LazyResettable<class_1921> renderType = new LazyResettable<>(() -> class_1921.method_24048("rei_cache", class_290.field_1585, class_293.class_5596.field_27382, 256,
            class_1921.class_4688.method_23598()
                    .method_34577(new class_4668.class_4683(cachedTextureLocation, class_9851.field_52396, false))
                    .method_34578(new class_4668.class_5942(class_10142.field_53879))
                    .method_23617(false)));
    
    public static class Sprite {
        public final float u0;
        public final float u1;
        public final float v0;
        public final float v1;
        
        public Sprite(float u0, float u1, float v0, float v1) {
            this.u0 = u0;
            this.u1 = u1;
            this.v0 = v0;
            this.v1 = v1;
        }
    }
    
    static {
        ReloadListenerRegistry.register(class_3264.field_14188, (barrier, resourceManager, preparationExecutor, reloadExecutor) -> {
            return barrier.method_18352(class_3902.field_17274).thenRunAsync(CachedEntryListRender::refresh, reloadExecutor);
        }, class_2960.method_60655("roughlyenoughitems", "cached_entries"));
    }
    
    public static void refresh() {
        if (ConfigObject.getInstance().doesCacheEntryRendering()) {
            InternalLogger.getInstance().info("Refreshing cached entry list texture...");
        }
        if (cachedTextureLocation != null) {
            class_310.method_1551().method_1531().method_4615(cachedTextureLocation);
            cachedTextureLocation = null;
            renderType.reset();
        }
        if (cachedTexture != null) {
            cachedTexture.close();
            cachedTexture = null;
        }
        hash = new Long2LongOpenHashMap();
    }
    
    @Nullable
    public static Sprite get(EntryStack<?> stack) {
        if (stack.getType() == VanillaEntryTypes.ITEM) {
            if (stack.getNullable(EntryStack.Settings.RENDERER) != null) {
                return null;
            }
            
            if (cachedTexture == null) {
                prepare();
            }
            
            long hashExact = EntryStacks.hashExact(stack);
            
            long hashOrDefault = hash.getOrDefault(hashExact, -1L);
            if (hashOrDefault != -1L) {
                // unpack
                int x = (int) (hashOrDefault >> 32);
                int y = (int) (hashOrDefault & 0xFFFFFFFFL);
                float width = cachedTexture.method_4525().method_4307();
                float height = cachedTexture.method_4525().method_4307();
                return new Sprite(x * RESOLUTION / width, (x + 1) * RESOLUTION / width, y * RESOLUTION / height, (y + 1) * RESOLUTION / height);
            }
        }
        
        return null;
    }
    
    private static void prepare() {
        int side = 4;
        List<EntryStack<?>> list = EntryRegistry.getInstance().getPreFilteredList();
        while (side * side < list.size()) {
            side++;
        }
        
        int width = side * RESOLUTION;
        int height = side * RESOLUTION;
        
        InternalLogger.getInstance().info("Preparing cached texture with size %sx%s for %sx%s entries", width, height, side, side);
        
        hash = new Long2LongOpenHashMap(list.size() + 10);
        class_310 minecraft = class_310.method_1551();
        class_6367 target = new class_6367(width, height, true);
        target.method_1236(0, 0, 0, 0);
        target.method_1235(true);
        Matrix4f projectionMatrix = new Matrix4f().setOrtho(0.0F, width, height, 0.0F, 1000.0F, 3000.0F);
        RenderSystem.setProjectionMatrix(projectionMatrix, class_10366.field_54954);
        Matrix4fStack modelViewStack = RenderSystem.getModelViewStack();
        modelViewStack.pushMatrix();
        modelViewStack.identity();
        modelViewStack.translate(0.0F, 0.0F, -2000.0F);
        
        class_308.method_24211();
        Rectangle bounds = new Rectangle();
        class_332 graphics = new class_332(minecraft, minecraft.method_22940().method_23000());
        
        int index = 0;
        for (EntryStack<?> stack : list) {
            int x = index % side;
            int y = index / side;
            bounds.setBounds(x * RESOLUTION, y * RESOLUTION, RESOLUTION, RESOLUTION);
            ((EntryRenderer<Object>) stack.getDefinition().getRenderer()).render((EntryStack<Object>) stack, graphics, bounds, -1, -1, 0);
            hash.put(EntryStacks.hashExact(stack), pack(x, y));
            index++;
        }
        
        class_1011 nativeImage = new class_1011(width, height, false);
        RenderSystem.bindTexture(target.method_30277());
        nativeImage.method_4327(0, false);
        nativeImage.method_4319();
        
        cachedTexture = new class_1043(nativeImage);
        cachedTextureLocation = class_2960.method_60655("roughlyenoughitems", "rei_cached_entries");
        minecraft.method_1531().method_4616(cachedTextureLocation, cachedTexture);
        renderType.reset();
        
        target.method_1238();
        // Minecraft.getInstance().levelRenderer.graphicsChanged();
        class_310.method_1551().method_1522().method_1235(true);
        
        modelViewStack.popMatrix();
    }
    
    private static long pack(int x, int y) {
        return ((long) x << 32) | (y & 0xFFFFFFFFL);
    }
}
