/*
 * This file is part of architectury.
 * Copyright (C) 2020, 2021, 2022 architectury
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

package dev.architectury.networking.forge;


import com.mojang.logging.LogUtils;
import dev.architectury.impl.NetworkAggregator;
import dev.architectury.networking.NetworkManager;
import dev.architectury.networking.NetworkManager.NetworkReceiver;
import dev.architectury.networking.SpawnEntityPacket;
import dev.architectury.networking.forge.client.ClientNetworkManagerImpl;
import dev.architectury.platform.hooks.EventBusesHooks;
import dev.architectury.utils.ArchitecturyConstants;
import dev.architectury.utils.Env;
import net.minecraft.core.RegistryAccess;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.network.protocol.Packet;
import net.minecraft.network.protocol.common.ClientboundCustomPayloadPacket;
import net.minecraft.network.protocol.common.ServerboundCustomPayloadPacket;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.network.protocol.game.ClientGamePacketListener;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerEntity;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.player.Player;
import net.neoforged.neoforge.network.event.RegisterPayloadHandlersEvent;
import net.neoforged.neoforge.network.handling.IPayloadContext;
import org.slf4j.Logger;

import static dev.architectury.networking.forge.client.ClientNetworkManagerImpl.getClientPlayer;
import static dev.architectury.networking.forge.client.ClientNetworkManagerImpl.getClientRegistryAccess;

public class NetworkManagerImpl {
    private static final Logger LOGGER = LogUtils.getLogger();
    
    public static NetworkAggregator.Adaptor getAdaptor() {
        return new NetworkAggregator.Adaptor() {
            @Override
            public <T extends CustomPacketPayload> void registerC2S(CustomPacketPayload.Type<T> type, StreamCodec<? super RegistryFriendlyByteBuf, T> codec, NetworkReceiver<T> receiver) {
                EventBusesHooks.whenAvailable(ArchitecturyConstants.MOD_ID, bus -> {
                    bus.<RegisterPayloadHandlersEvent>addListener(event -> {
                        event.registrar(type.id().getNamespace()).optional().playToServer(type, codec, (arg, context) -> {
                            receiver.receive(arg, context(context.player(), context, false));
                        });
                    });
                });
            }
            
            @Override
            public <T extends CustomPacketPayload> void registerS2C(CustomPacketPayload.Type<T> type, StreamCodec<? super RegistryFriendlyByteBuf, T> codec, NetworkReceiver<T> receiver) {
                EventBusesHooks.whenAvailable(ArchitecturyConstants.MOD_ID, bus -> {
                    bus.<RegisterPayloadHandlersEvent>addListener(event -> {
                        event.registrar(type.id().getNamespace()).optional().playToClient(type, codec, (arg, context) -> {
                            receiver.receive(arg, context(context.player(), context, true));
                        });
                    });
                });
            }
            
            @Override
            public <T extends CustomPacketPayload> Packet<?> toC2SPacket(T payload) {
                return new ServerboundCustomPayloadPacket(payload);
            }
            
            @Override
            public <T extends CustomPacketPayload> Packet<?> toS2CPacket(T payload) {
                return new ClientboundCustomPayloadPacket(payload);
            }
            
            @Override
            public <T extends CustomPacketPayload> void registerS2CType(CustomPacketPayload.Type<T> type, StreamCodec<? super RegistryFriendlyByteBuf, T> codec) {
                registerS2C(type, codec, (payload, context) -> {
                });
            }
            
            public NetworkManager.PacketContext context(Player player, IPayloadContext taskQueue, boolean client) {
                return new NetworkManager.PacketContext() {
                    @Override
                    public Player getPlayer() {
                        return getEnvironment() == Env.CLIENT ? getClientPlayer() : player;
                    }
                    
                    @Override
                    public void queue(Runnable runnable) {
                        taskQueue.enqueueWork(runnable);
                    }
                    
                    @Override
                    public Env getEnvironment() {
                        return client ? Env.CLIENT : Env.SERVER;
                    }
                    
                    @Override
                    public RegistryAccess registryAccess() {
                        return getEnvironment() == Env.CLIENT ? getClientRegistryAccess() : player.registryAccess();
                    }
                };
            }
        };
    }
    
    public static boolean canServerReceive(ResourceLocation id) {
        return ClientNetworkManagerImpl.canServerReceive(id);
    }
    
    public static boolean canPlayerReceive(ServerPlayer player, ResourceLocation id) {
        return player.connection.hasChannel(id);
    }
    
    public static <T extends CustomPacketPayload> void sendToServer(T payload) {
        ClientNetworkManagerImpl.sendToServer(payload);
    }
    
    public static Packet<ClientGamePacketListener> createAddEntityPacket(Entity entity, ServerEntity serverEntity) {
        return SpawnEntityPacket.create(entity, serverEntity);
    }
}
