/*
 * This file is part of fabric-loom, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2022-2025 FabricMC
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package dev.architectury.loom.mcpconfig.steplogic;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;

import dev.architectury.loom.forge.config.ConfigValue;
import dev.architectury.loom.forge.tool.ForgeToolExecutor;
import dev.architectury.loom.util.collection.CollectionUtil;
import org.gradle.api.Action;
import org.gradle.api.Project;
import org.gradle.api.file.FileCollection;
import org.gradle.api.logging.Logger;
import org.gradle.api.provider.Provider;
import org.jetbrains.annotations.Nullable;

import net.fabricmc.loom.util.download.DownloadBuilder;
import net.fabricmc.loom.util.service.Service;
import net.fabricmc.loom.util.service.ServiceFactory;

/**
 * The logic for executing a step. This corresponds to the {@code type} key in the step JSON format.
 */
public abstract class StepLogic<O extends Service.Options> extends Service<O> {
	public StepLogic(O options, ServiceFactory serviceFactory) {
		super(options, serviceFactory);
	}

	public abstract void execute(ExecutionContext context) throws IOException;

	public String getDisplayName(String stepName) {
		return stepName;
	}

	public interface ExecutionContext {
		Logger logger();
		Path setOutput(String fileName) throws IOException;
		Path setOutput(Path output);
		Path cache() throws IOException;
		/** Mappings extracted from {@code data.mappings} in the MCPConfig JSON. */
		Path mappings();
		String resolve(ConfigValue value);
		DownloadBuilder downloadBuilder(String url);
		void javaexec(Action<? super ForgeToolExecutor.Settings> configurator);

		default List<String> resolve(List<ConfigValue> configValues) {
			return CollectionUtil.map(configValues, this::resolve);
		}
	}

	public interface SetupContext {
		Project project();
		Path downloadFile(String url) throws IOException;
		Path downloadDependency(String notation);
		Provider<FileCollection> getMinecraftLibraries();
	}

	@FunctionalInterface
	public interface StepLogicProvider {
		@Nullable Provider<? extends Service.Options> getStepLogic(SetupContext context, String name, String type);
	}
}
