/*
 * Decompiled with CFR 0.152.
 */
package net.fabricmc.loom.util.download;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Locale;
import net.fabricmc.loom.util.download.Download;
import net.fabricmc.loom.util.download.DownloadException;
import net.fabricmc.loom.util.download.DownloadExecutor;
import net.fabricmc.loom.util.download.DownloadProgressListener;

public class DownloadBuilder {
    private static final Duration ONE_DAY = Duration.ofDays(1L);
    private final URI url;
    private String expectedHash = null;
    private boolean useEtag = true;
    private boolean forceDownload = false;
    private boolean offline = false;
    private Duration maxAge = Duration.ZERO;
    private DownloadProgressListener progressListener = DownloadProgressListener.NONE;
    private int maxRetries = 3;
    private boolean allowInsecureProtocol = false;
    private HttpClient.Version httpVersion = HttpClient.Version.HTTP_2;

    private DownloadBuilder(URI url) {
        this.url = url;
    }

    static DownloadBuilder create(String url) throws URISyntaxException {
        return new DownloadBuilder(new URI(url));
    }

    public DownloadBuilder sha1(String sha1) {
        this.expectedHash = "sha1:" + sha1;
        return this;
    }

    public DownloadBuilder etag(boolean useEtag) {
        this.useEtag = useEtag;
        return this;
    }

    public DownloadBuilder forceDownload() {
        this.forceDownload = true;
        return this;
    }

    public DownloadBuilder offline() {
        this.offline = true;
        return this;
    }

    public DownloadBuilder maxAge(Duration duration) {
        this.maxAge = duration;
        return this;
    }

    public DownloadBuilder progress(DownloadProgressListener progressListener) {
        this.progressListener = progressListener;
        return this;
    }

    public DownloadBuilder maxRetries(int maxRetries) {
        this.maxRetries = maxRetries;
        return this;
    }

    public DownloadBuilder defaultCache() {
        this.etag(true);
        return this.maxAge(ONE_DAY);
    }

    public DownloadBuilder allowInsecureProtocol() {
        this.allowInsecureProtocol = true;
        return this;
    }

    public DownloadBuilder httpVersion(HttpClient.Version httpVersion) {
        this.httpVersion = httpVersion;
        return this;
    }

    private Download build(int downloadAttempt) {
        if (!this.allowInsecureProtocol && !DownloadBuilder.isSecureUrl(this.url)) {
            throw new IllegalArgumentException("Cannot create download for url (%s) with insecure protocol".formatted(this.url.toString()));
        }
        return new Download(this.url, this.expectedHash, this.useEtag, this.forceDownload, this.offline, this.maxAge, this.progressListener, this.httpVersion, downloadAttempt);
    }

    public void downloadPathAsync(Path path, DownloadExecutor executor) {
        executor.runAsync(() -> this.downloadPath(path));
    }

    public void downloadPath(Path path) throws DownloadException {
        this.withRetries(download -> {
            download.downloadPath(path);
            return null;
        });
    }

    public String downloadString() throws DownloadException {
        return this.withRetries(Download::downloadString);
    }

    public String downloadString(Path cache) throws DownloadException {
        return this.withRetries(download -> {
            download.downloadPath(cache);
            try {
                return Files.readString(cache, StandardCharsets.UTF_8);
            }
            catch (IOException e) {
                try {
                    Files.deleteIfExists(cache);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                throw new DownloadException("Failed to download and read string", e);
            }
        });
    }

    private <T> T withRetries(DownloadFunction<T> supplier) throws DownloadException {
        for (int i = 1; i <= this.maxRetries; ++i) {
            try {
                if (i == this.maxRetries) {
                    this.httpVersion(HttpClient.Version.HTTP_1_1);
                }
                return supplier.get(this.build(i));
            }
            catch (DownloadException e) {
                if (e.getStatusCode() == 404) {
                    throw e;
                }
                if (i != this.maxRetries) continue;
                throw new DownloadException(String.format(Locale.ENGLISH, "Failed download after %d attempts", this.maxRetries), e);
            }
        }
        throw new IllegalStateException();
    }

    private static boolean isSecureUrl(URI url) {
        if ("127.0.0.1".equals(url.getHost())) {
            return true;
        }
        String scheme = url.getScheme();
        return !"http".equalsIgnoreCase(scheme);
    }

    @FunctionalInterface
    private static interface DownloadFunction<T> {
        public T get(Download var1) throws DownloadException;
    }
}

