/*
 * This file is part of architectury.
 * Copyright (C) 2020, 2021, 2022 architectury
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

package dev.architectury.registry.registries.fabric;

import com.google.common.base.Objects;
import com.google.common.base.Suppliers;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import dev.architectury.impl.RegistrySupplierImpl;
import dev.architectury.registry.registries.Registrar;
import dev.architectury.registry.registries.RegistrarBuilder;
import dev.architectury.registry.registries.RegistrarManager;
import dev.architectury.registry.registries.RegistrySupplier;
import dev.architectury.registry.registries.options.DefaultIdRegistrarOption;
import dev.architectury.registry.registries.options.RegistrarOption;
import dev.architectury.registry.registries.options.StandardRegistrarOption;
import net.fabricmc.fabric.api.event.registry.FabricRegistryBuilder;
import net.fabricmc.fabric.api.event.registry.RegistryAttribute;
import net.fabricmc.fabric.api.event.registry.RegistryEntryAddedCallback;
import net.minecraft.class_2370;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Supplier;

public class RegistrarManagerImpl {
    private static final Multimap<RegistryEntryId<?>, Consumer<?>> LISTENERS = HashMultimap.create();
    private static final Set<class_5321<?>> LISTENED_REGISTRIES = new HashSet<>();
    
    private static void listen(class_5321<?> resourceKey, class_2960 id, Consumer<?> listener) {
        if (LISTENED_REGISTRIES.add(resourceKey)) {
            class_2378<?> registry = java.util.Objects.requireNonNull(class_7923.field_41167.method_63535(resourceKey.method_29177()), "Registry " + resourceKey + " not found!");
            RegistryEntryAddedCallback.event(registry).register((rawId, entryId, object) -> {
                RegistryEntryId<?> registryEntryId = new RegistryEntryId<>(resourceKey, entryId);
                for (Consumer<?> consumer : LISTENERS.get(registryEntryId)) {
                    ((Consumer<Object>) consumer).accept(object);
                }
                LISTENERS.removeAll(registryEntryId);
            });
        }
        
        LISTENERS.put(new RegistryEntryId<>(resourceKey, id), listener);
    }
    
    public static RegistrarManager.RegistryProvider _get(String modId) {
        return new RegistryProviderImpl(modId);
    }
    
    public static class RegistryProviderImpl implements RegistrarManager.RegistryProvider {
        private final String modId;
        
        public RegistryProviderImpl(String modId) {
            this.modId = modId;
        }
        
        @Override
        public <T> Registrar<T> get(class_5321<class_2378<T>> key) {
            return new RegistrarImpl<>(modId, (class_2378<T>) java.util.Objects.requireNonNull(class_7923.field_41167.method_63535(key.method_29177()), "Registry " + key + " not found!"));
        }
        
        @Override
        public <T> Registrar<T> get(class_2378<T> registry) {
            return new RegistrarImpl<>(modId, registry);
        }
        
        @Override
        public <T> void forRegistry(class_5321<class_2378<T>> key, Consumer<Registrar<T>> consumer) {
            consumer.accept(get(key));
        }
        
        @Override
        public <T> RegistrarBuilder<T> builder(Class<T> type, class_2960 registryId) {
            return new RegistrarBuilderWrapper<>(modId, type, registryId);
        }
    }
    
    public static class RegistryEntryId<T> {
        private final class_5321<T> registryKey;
        private final class_2960 id;
        
        public RegistryEntryId(class_5321<T> registryKey, class_2960 id) {
            this.registryKey = registryKey;
            this.id = id;
        }
        
        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof RegistryEntryId)) return false;
            RegistryEntryId<?> that = (RegistryEntryId<?>) o;
            return java.util.Objects.equals(registryKey, that.registryKey) && java.util.Objects.equals(id, that.id);
        }
        
        @Override
        public int hashCode() {
            return java.util.Objects.hash(registryKey, id);
        }
    }
    
    public static class RegistrarBuilderWrapper<T> implements RegistrarBuilder<T> {
        private final String modId;
        private final Class<T> type;
        private final class_2960 registryId;
        private final List<Consumer<FabricRegistryBuilder<T, ? extends class_2370<T>>>> apply = new ArrayList<>();
        @Nullable
        private class_2960 defaultId;
        
        public RegistrarBuilderWrapper(String modId, Class<T> type, class_2960 registryId) {
            this.modId = modId;
            this.type = type;
            this.registryId = registryId;
        }
        
        @Override
        public Registrar<T> build() {
            final var builder = defaultId == null
                    ? FabricRegistryBuilder.createSimple(type, registryId)
                    : FabricRegistryBuilder.createDefaulted(type, registryId, defaultId);
            apply.forEach(consumer -> consumer.accept(builder));
            return RegistrarManager.get(modId).get(builder.buildAndRegister());
        }
        
        @Override
        public RegistrarBuilder<T> option(RegistrarOption option) {
            if (option == StandardRegistrarOption.SYNC_TO_CLIENTS) {
                this.apply.add(builder -> builder.attribute(RegistryAttribute.SYNCED));
            } else if (option instanceof DefaultIdRegistrarOption opt) {
                this.defaultId = opt.defaultId();
            }
            return this;
        }
    }
    
    public static class RegistrarImpl<T> implements Registrar<T> {
        private final String modId;
        private class_2378<T> delegate;
        
        public RegistrarImpl(String modId, class_2378<T> delegate) {
            this.modId = modId;
            this.delegate = delegate;
        }
        
        @Override
        public RegistrySupplier<T> delegate(class_2960 id) {
            Supplier<T> value = Suppliers.memoize(() -> get(id));
            RegistrarImpl<T> registrar = this;
            return new RegistrySupplierImpl<T>() {
                @Nullable
                class_6880<T> holder = null;
                
                @Nullable
                @Override
                public class_6880<T> getHolder() {
                    if (holder != null) return holder;
                    return holder = registrar.getHolder(getId());
                }
                
                @Override
                public RegistrarManager getRegistrarManager() {
                    return RegistrarManager.get(modId);
                }
                
                @Override
                public Registrar<T> getRegistrar() {
                    return registrar;
                }
                
                @Override
                public class_2960 getRegistryId() {
                    return delegate.method_46765().method_29177();
                }
                
                @Override
                public class_2960 getId() {
                    return id;
                }
                
                @Override
                public boolean isPresent() {
                    return contains(id);
                }
                
                @Override
                public T get() {
                    return value.get();
                }
                
                @Override
                public int hashCode() {
                    return Objects.hashCode(getRegistryId(), getId());
                }
                
                @Override
                public boolean equals(Object obj) {
                    if (this == obj) return true;
                    if (!(obj instanceof RegistrySupplier<?> other)) return false;
                    return other.getRegistryId().equals(getRegistryId()) && other.getId().equals(getId());
                }
                
                @Override
                public String toString() {
                    return getRegistryId().toString() + "@" + id.toString();
                }
            };
        }
        
        @Override
        public <E extends T> RegistrySupplier<E> register(class_2960 id, Supplier<E> supplier) {
            class_2378.method_10230(delegate, id, supplier.get());
            return (RegistrySupplier<E>) delegate(id);
        }
        
        @Override
        public @Nullable class_2960 getId(T obj) {
            return delegate.method_10221(obj);
        }
        
        @Override
        public int getRawId(T obj) {
            return delegate.method_10206(obj);
        }
        
        @Override
        public Optional<class_5321<T>> getKey(T obj) {
            return delegate.method_29113(obj);
        }
        
        @Override
        public @Nullable T get(class_2960 id) {
            return delegate.method_63535(id);
        }
        
        @Override
        public T byRawId(int rawId) {
            return delegate.method_10200(rawId);
        }
        
        @Override
        public boolean contains(class_2960 id) {
            return delegate.method_10235().contains(id);
        }
        
        @Override
        public boolean containsValue(T obj) {
            return delegate.method_29113(obj).isPresent();
        }
        
        @Override
        public Set<class_2960> getIds() {
            return delegate.method_10235();
        }
        
        @Override
        public Set<Map.Entry<class_5321<T>, T>> entrySet() {
            return delegate.method_29722();
        }
        
        @Override
        public class_5321<? extends class_2378<T>> key() {
            return delegate.method_46765();
        }
        
        @Override
        @Nullable
        public class_6880<T> getHolder(class_5321<T> key) {
            return delegate.method_46746(key).orElse(null);
        }
        
        @Override
        public Iterator<T> iterator() {
            return delegate.iterator();
        }
        
        @Override
        public void listen(class_2960 id, Consumer<T> callback) {
            if (contains(id)) {
                callback.accept(get(id));
            } else {
                RegistrarManagerImpl.listen(key(), id, callback);
            }
        }
    }
}
