/*
 * This file is part of architectury.
 * Copyright (C) 2020, 2021, 2022 architectury
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

package dev.architectury.networking.fabric;

import com.mojang.logging.LogUtils;
import dev.architectury.impl.NetworkAggregator;
import dev.architectury.networking.NetworkManager;
import dev.architectury.networking.NetworkManager.NetworkReceiver;
import dev.architectury.networking.SpawnEntityPacket;
import dev.architectury.networking.client.fabric.ClientNetworkManagerImpl;
import dev.architectury.networking.transformers.PacketSink;
import dev.architectury.platform.Platform;
import dev.architectury.utils.Env;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayNetworking;
import net.fabricmc.fabric.api.networking.v1.PayloadTypeRegistry;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_1255;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_2596;
import net.minecraft.class_2602;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3222;
import net.minecraft.class_3231;
import net.minecraft.class_5455;
import net.minecraft.class_634;
import net.minecraft.class_8710;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import org.slf4j.Logger;

public class NetworkManagerImpl {
    private static final Logger LOGGER = LogUtils.getLogger();
    
    public static NetworkAggregator.Adaptor getAdaptor() {
        return new NetworkAggregator.Adaptor() {
            @Override
            public <T extends class_8710> void registerC2S(class_8710.class_9154<T> type, class_9139<? super class_9129, T> codec, NetworkReceiver<T> receiver) {
                LOGGER.info("Registering C2S receiver with id {}", type.comp_2242());
                PayloadTypeRegistry.playC2S().register(type, codec);
                ServerPlayNetworking.registerGlobalReceiver(type, (payload, fabricContext) -> {
                    var context = context(fabricContext.player(), fabricContext.server(), false);
                    receiver.receive(payload, context);
                });
            }
            
            @Override
            public <T extends class_8710> void registerS2C(class_8710.class_9154<T> type, class_9139<? super class_9129, T> codec, NetworkReceiver<T> receiver) {
                PayloadTypeRegistry.playS2C().register(type, codec);
                if (Platform.getEnvironment() == Env.CLIENT)
                    ClientNetworkManagerImpl.registerS2C(type, codec, receiver);
            }
            
            @Override
            public <T extends class_8710> class_2596<?> toC2SPacket(T payload) {
                return ClientPlayNetworking.createC2SPacket(payload);
            }
            
            @Override
            public <T extends class_8710> class_2596<?> toS2CPacket(T payload) {
                return ServerPlayNetworking.createS2CPacket(payload);
            }
            
            @Override
            public <T extends class_8710> void registerS2CType(class_8710.class_9154<T> type, class_9139<? super class_9129, T> codec) {
                PayloadTypeRegistry.playS2C().register(type, codec);
            }
        };
    }
    
    public static NetworkManager.PacketContext context(class_1657 player, class_1255<?> taskQueue, boolean client) {
        return new NetworkManager.PacketContext() {
            @Override
            public class_1657 getPlayer() {
                return player;
            }
            
            @Override
            public void queue(Runnable runnable) {
                taskQueue.execute(runnable);
            }
            
            @Override
            public Env getEnvironment() {
                return client ? Env.CLIENT : Env.SERVER;
            }
            
            @Override
            public class_5455 registryAccess() {
                return player.method_56673();
            }
        };
    }
    
    @Environment(EnvType.CLIENT)
    public static boolean canServerReceive(class_2960 id) {
        return ClientPlayNetworking.canSend(id);
    }
    
    public static boolean canPlayerReceive(class_3222 player, class_2960 id) {
        return ServerPlayNetworking.canSend(player, id);
    }
    
    public static <T extends class_8710> void sendToServer(T payload) {
        class_634 connection = class_310.method_1551().method_1562();
        if (connection == null) return;
        NetworkManager.collectPackets(PacketSink.client(), NetworkManager.clientToServer(), payload, connection.method_29091());
    }
    
    public static class_2596<class_2602> createAddEntityPacket(class_1297 entity, class_3231 serverEntity) {
        return SpawnEntityPacket.create(entity, serverEntity);
    }
}
